/*******************************************************************************
 * Copyright (c) 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.AsyncBarrierImpl;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.AsyncMultiProcedure;

public class BlockingAsyncMultiProcedure<Result> implements AsyncMultiProcedure<Result> {

    private static final Object NO_RESULT = new Object();

    private final Object key;
    private final AsyncBarrierImpl barrier;
    private final ReadGraphImpl procedureGraph;
    private final AsyncMultiProcedure<Result> procedure;

    private Object result = NO_RESULT;
    private Throwable exception = null;

    public BlockingAsyncMultiProcedure(AsyncBarrierImpl barrier, ReadGraphImpl procedureGraph, AsyncMultiProcedure<Result> procedure, Object key) {
        this.procedure = procedure;
        this.key = key;
        this.barrier = barrier;
        this.barrier.inc();
        this.procedureGraph = procedureGraph;
    }

    @Override
    public void execute(AsyncReadGraph graph, Result result) {
        this.result = result;
        try {
            if(procedure != null) procedure.execute(procedureGraph, result);
        } catch (Throwable throwable) {
            Logger.defaultLogError("AsyncProcedure.execute threw for " + procedure, throwable);
        }
    }

    @Override
    public void finished(AsyncReadGraph graph) {
        try {
            if(procedure != null) procedure.finished(procedureGraph);
        } catch (Throwable throwable) {
            Logger.defaultLogError("AsyncProcedure.finish threw for " + procedure, throwable);
        } finally {
            barrier.dec();
        }
    }

    @Override
    public void exception(AsyncReadGraph graph, Throwable t) {
        this.exception = t;
        try {
            if (procedure != null) procedure.exception(procedureGraph, t);
        } catch (Throwable throwable) {
            Logger.defaultLogError("AsyncProcedure.exception threw for " + procedure, throwable);
        } finally {
            barrier.dec();
        }
    }

    @SuppressWarnings("unchecked")
    public Result get() throws DatabaseException {
        
        barrier.waitBarrier(key, procedureGraph);
        
        if (exception != null) {
            if (exception instanceof DatabaseException) throw (DatabaseException) exception;
            throw new DatabaseException(exception);
        } else {
            return (Result) result;
        }
        
    }

    @SuppressWarnings("unchecked")
    public Result getResult() {
        return (Result) result;
    }

    public Throwable getException() {
        return exception;
    }

    @Override
    public String toString() {
        return "." + procedure; 
    }

}
