package org.simantics.db.common.utils;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;

public class Versions {

	public static String make(String baseName, String version) {
		return baseName + "@" + version;
	}
	
	public static String getBaseVersion(String version) {
		try {
			Integer.parseInt(version);
			return "1";
		} catch (NumberFormatException e) {
		}
		return "A";
	}
	
	public static String getBaseName(String name) {
		int pos = name.lastIndexOf('@');
		if(pos == -1) return name;
		return name.substring(0, pos);
	}

	public static String getVersion(String name) {
    	if(name == null) return null;
    	if(!name.contains("@")) return null;
		int pos = name.lastIndexOf('@');
		return name.substring(pos+1);
	}

	public static String getBaseName(ReadGraph graph, Resource artifact) throws DatabaseException {
    	Layer0 L0 = Layer0.getInstance(graph);
    	String name = graph.getPossibleRelatedValue(artifact, L0.HasName, Bindings.STRING);
    	if(name == null) return "";
    	if(!name.contains("@")) return name;
		int pos = name.lastIndexOf('@');
		return name.substring(0, pos);
	}

	public static String getVersion(ReadGraph graph, Resource artifact) throws DatabaseException {
    	Layer0 L0 = Layer0.getInstance(graph);
    	String name = graph.getPossibleRelatedValue(artifact, L0.HasName, Bindings.STRING);
    	return getVersion(name);
	}

	public static Collection<NamedResource> getOlderVersions(ReadGraph graph, Resource artifact) throws DatabaseException {

		Set<NamedResource> result = new HashSet<NamedResource>();
		
		VersionInfo info = graph.syncRequest(new VersionInfoRequest(artifact));
		result.addAll(info.getOlderVersions());
		
		Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(artifact));
		if(indexRoot == null || indexRoot.equals(artifact)) return result;

		VersionInfo versions = graph.syncRequest(new VersionInfoRequest(indexRoot));
		for(NamedResource ontology : versions.getOlderVersions()) {
			VersionMap map = match(graph, artifact, ontology.getResource());
			result.addAll(map.getOlderOrEqualVersions(graph, artifact));
		}
		
		return result;
		
	}
	
	public static Collection<NamedResource> getNewerVersions(ReadGraph graph, Resource artifact) throws DatabaseException {

		Set<NamedResource> result = new HashSet<NamedResource>();
		
		VersionInfo info = graph.syncRequest(new VersionInfoRequest(artifact));
		result.addAll(info.getNewerVersions());
		
		Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(artifact));
		if(indexRoot == null || indexRoot.equals(artifact)) return result;
		
		VersionInfo versions = graph.syncRequest(new VersionInfoRequest(indexRoot));
		for(NamedResource ontology : versions.getNewerVersions()) {
			VersionMap map = match(graph, artifact, ontology.getResource());
			result.addAll(map.getNewerOrEqualVersions(graph, artifact));
		}

		return result;
		
	}

	public static VersionMap match(ReadGraph graph, Resource type, Resource newOntology) throws DatabaseException {
    	Resource originalOntology = graph.syncRequest(new PossibleIndexRoot(type));
    	String originalURI = graph.getURI(originalOntology);
    	String newOntologyURI = graph.getURI(newOntology);
    	String typeURI = graph.getURI(type);
    	String typeURIInNewOntology = newOntologyURI + typeURI.substring(originalURI.length());
    	String[] parts = URIStringUtils.splitURI(typeURIInNewOntology); 
    	Resource newLibrary = graph.getPossibleResource(parts[0]);
    	if(newLibrary == null) return null;
    	return graph.syncRequest(new VersionMapRequest(newLibrary));
    }

    public static String getStandardNameString(ReadGraph graph, Resource r) throws DatabaseException {
    	
    	Layer0 L0 = Layer0.getInstance(graph);
    	String name = graph.getPossibleRelatedValue(r, L0.HasName, Bindings.STRING);
    	if(name != null) {
    		VersionInfo versions = graph.syncRequest(new VersionInfoRequest(r));
    		int nVersions = versions.versions.size();
    		// No version display, if "A", "1" or "" is the only version present
    		if ("".equals(versions.version))
    			return nVersions == 1 ? name : name + " [No version]";
    		String base = versions.baseName;
    		String version = versions.version;
    		if (("A".equals(version) || "1".equals(version)) && nVersions == 1)
    			return base;
    		else
    			return base + " [Version " + version + "]";
    	}
    	// Fallback logic
    	return NameUtils.getSafeName(graph, r);
    	
    }
    
    public static String getStandardPathNameString(ReadGraph graph, Resource r) throws DatabaseException {
        Layer0 L0 = graph.l0();

        String standardName = getStandardNameString(graph, r); 

        Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(r));
        if(indexRoot.equals(r))
            return standardName;

        Resource indexRootParent = graph.getPossibleObject(indexRoot, L0.PartOf);
        if (indexRootParent == null)
            return standardName;

        String indexRootName = getStandardNameString(graph, indexRoot);

        String rootLibraryUri = graph.getURI(graph.getRootLibrary());
        String indexRootParentURI = graph.getPossibleURI(indexRootParent);
        String indexRootParentPath = indexRootParentURI != null
                ? indexRootParentURI.substring(rootLibraryUri.length())
                : "";

        Resource parent = graph.getPossibleObject(r, L0.PartOf);
        String pathInIndexRoot = "";
        if (parent != null && !parent.equals(indexRoot)) {
            String indexRootURI = graph.getPossibleURI(indexRoot);
            String parentURI = parent != null ? graph.getURI(parent) : "";
            pathInIndexRoot = indexRootURI != null
                    ? parentURI.substring(indexRootURI.length()+1) + "/"
                    : "";
        }

        return URIStringUtils.unescape(pathInIndexRoot) + standardName
                + "  in  "
                + URIStringUtils.unescape(indexRootParentPath) + '/' + indexRootName;
    }

}
