package org.simantics.db.common;

import java.util.HashSet;
import java.util.Set;

import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.Binding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.graph.representation.TransferableGraphUtils;
import org.simantics.graph.representation.Value;
import org.simantics.layer0.Layer0;

public class TransferableGraphRequests {
	
	private TransferableGraph1 tg;
	private ReadGraph graph;
	
	public TransferableGraphRequests(TransferableGraph1 tg, ReadGraph graph) {
		this.tg = tg;
		this.graph = graph;
	}

	public Resource getExternal(int id) throws DatabaseException {
		String uri = TransferableGraphUtils.getURI(tg, id);
		return graph.getPossibleResource(uri);
	}
	
	public Set<Integer> getObjects(int subject, Resource relation) throws DatabaseException {

		Set<Integer> result = new HashSet<Integer>();
		for(int i=0;i<tg.statements.length;i+=4) {
			if(tg.statements[i] == subject) {
				Resource predicate = getExternal(tg.statements[i+1]);
				if(predicate != null)
					if(graph.isSubrelationOf(predicate, relation)) result.add(tg.statements[i+3]); 
			}
		}
		return result;
		
	}

	public Set<Integer> getSubjects(int object, Resource inverseRelation) throws DatabaseException {

		Set<Integer> result = new HashSet<Integer>();
		for(int i=0;i<tg.statements.length;i+=4) {
			if(tg.statements[i+3] == object) {
				Resource predicate = getExternal(tg.statements[i+2]);
				if(predicate != null)
					if(graph.isSubrelationOf(predicate, inverseRelation)) result.add(tg.statements[i]); 
			}
		}
		return result;
		
	}

	public int getPossibleObject(int subject, Resource relation) throws DatabaseException {
		Set<Integer> objects = getObjects(subject, relation);
		if(objects.size() == 1) return objects.iterator().next();
		return -1;
	}

	public int getSingleObject(int subject, Resource relation) throws DatabaseException {

		Set<Integer> objects = getObjects(subject, relation);
		if(objects.size() == 1) return objects.iterator().next();
		else if (objects.size() == 0) throw new DatabaseException("No objects for subject " + subject + " and relation " + graph.getURI(relation));
		else throw new DatabaseException("Multiple (" + objects.size() + ") objects for subject " + subject + " and relation " + graph.getURI(relation));
		
	}

	public int getPossibleSubject(int object, Resource inverseRelation) throws DatabaseException {
		Set<Integer> subjects = getSubjects(object, inverseRelation);
		if(subjects.size() == 1) return subjects.iterator().next();
		return -1;
	}

	public int getSingleSubject(int object, Resource inverseRelation) throws DatabaseException {

		Set<Integer> subjects = getObjects(object, inverseRelation);
		if(subjects.size() == 1) return subjects.iterator().next();
		else if (subjects.size() == 0) throw new DatabaseException("No subjects for object " + object + " and relation " + graph.getURI(inverseRelation));
		else throw new DatabaseException("Multiple (" + subjects.size() + ") subjects for object " + object + " and relation " + graph.getURI(inverseRelation));
		
	}

	public <T> T getRelatedValue(int subject, Resource relation, Binding binding) throws DatabaseException {
		
		int object = getSingleObject(subject, relation);
		return getValue(object, binding);
		
	}

	public <T> T getPossibleRelatedValue(int subject, Resource relation, Binding binding) throws DatabaseException {
		int object = getPossibleObject(subject, relation);
		if (object == -1)
			return null;
		return getPossibleValue(object, binding);
	}

	@SuppressWarnings("unchecked")
	public <T> T getValue(int subject, Binding binding) throws DatabaseException {

		Value value = TransferableGraphUtils.findValue(tg, subject);
		if(value == null) throw new DatabaseException("No value for subject " + subject);
		try {
			return (T)value.value.getValue(binding);
		} catch (AdaptException e) {
			throw new DatabaseException(e);
		}
		
	}

	@SuppressWarnings("unchecked")
	public <T> T getPossibleValue(int subject, Binding binding) {
		Value value = TransferableGraphUtils.findValue(tg, subject);
		if(value == null) return null;
		try {
			return (T)value.value.getValue(binding);
		} catch (AdaptException e) {
			return null;
		}
	}

	public boolean isInstanceOf(int subject, Resource type) throws DatabaseException {

		Layer0 L0 = Layer0.getInstance(graph);
		for(int t : getObjects(subject, L0.InstanceOf)) {
			Resource tr = getExternal(t);
			if(tr != null)
				if(graph.isInheritedFrom(tr, type)) return true;
		}
		
		return false;
		
	}
	

	public String formatResource(int r) throws DatabaseException {
		
		String result = "" + r;
		String uri = TransferableGraphUtils.getURI(tg, r);
		if(!uri.isEmpty()) result = uri;
		
		Value value = TransferableGraphUtils.findValue(tg, r);
		if(value != null) result += " (value " + value.value + ")";
		
		return result;
		
	}
	
	public String listStatements() throws DatabaseException {
		
		StringBuilder b = new StringBuilder();
    	for(int i=0;i<tg.statements.length;i+=4) {
    		int subject = tg.statements[i];
    		int predicate = tg.statements[i+1];
    		int object = tg.statements[i+3];
    		b.append(formatResource(subject) + " " + formatResource(predicate) + " " + formatResource(object) + "\n");
    	}
    	return b.toString();
		
	}
	
	public ReadGraph getGraph() {
		return graph;
	}
	
}
