/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.request;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.AsyncMultiProcedureAdapter;
import org.simantics.db.common.procedure.adapter.AsyncProcedureAdapter;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;

/**
 * Asynchronous version of {@link MappedParts}.
 * 
 * @author Tuukka Lehtonen
 */
public class AsyncMappedParts extends ResourceAsyncRead<Map<Resource, Resource>> {

    public AsyncMappedParts(Resource flat) {
        super(flat);
    }

    @Override
    public void perform(AsyncReadGraph graph, final AsyncProcedure<Map<Resource, Resource>> procedure) {

        final AtomicInteger done = new AtomicInteger(1);
        final Layer0 l0 = graph.getService(Layer0.class);
        final Layer0X L0X = graph.getService(Layer0X.class);
        final Map<Resource, Resource> result = new ConcurrentHashMap<Resource, Resource>();

        graph.forEachObject(resource, l0.ConsistsOf, new AsyncMultiProcedureAdapter<Resource>() {
            @Override
            public void execute(AsyncReadGraph graph, final Resource part) {
                done.incrementAndGet();
                graph.forPossibleObject(part, L0X.Represents, new AsyncProcedureAdapter<Resource>() {
                    @Override
                    public void execute(AsyncReadGraph graph, Resource represents) {
                        if (represents != null)
                            result.put(represents, part);
                        if(done.decrementAndGet() == 0) {
                            procedure.execute(graph, result);
                        }
                    }
                });
            }
            @Override
            public void exception(AsyncReadGraph graph, Throwable t) {
                procedure.exception(graph, t);
            }
            @Override
            public void finished(AsyncReadGraph graph) {
                if(done.decrementAndGet() == 0) {
                    procedure.execute(graph, result);
                }
            }
        });
    }

}
