package org.simantics.db.common.request;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.AdaptionException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.exception.ValidationException;

/**
 * @author Tuukka Lehtonen
 * 
 * @param <T> the adaption target interface
 */
public class Adapt<T> extends ResourceRead<T> {

    final protected Class<T> target;
    final protected boolean  allowNull;
    final protected boolean  uniqueResult;

    public Adapt(Resource resource, Class<T> target) {
        this(resource, target, false, false);
    }

    public Adapt(Resource resource, Class<T> target, boolean allowNull) {
        this(resource, target, allowNull, false);
    }

    public Adapt(Resource resource, Class<T> target, boolean allowNull, boolean uniqueResult) {
        super(resource);
        assert target != null;
        this.target = target;
        this.allowNull = allowNull;
        this.uniqueResult = uniqueResult;
    }

    @Override
    public T perform(ReadGraph graph) throws AdaptionException, ValidationException, ServiceException {
        if (allowNull) {
            if (uniqueResult)
                return graph.getPossibleUniqueAdapter(resource, target);
            return graph.getPossibleAdapter(resource, target);
        }
        if (uniqueResult)
            return graph.adaptUnique(resource, target);
        return graph.adapt(resource, target);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = super.hashCode();
        result = prime * result + (allowNull ? 1231 : 1237);
        result = prime * result + (target.hashCode());
        result = prime * result + (uniqueResult ? 1303 : 1319);
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null)
            return false;
        if (this == obj)
            return true;
        if (getClass() != obj.getClass())
            return false;
        Adapt<?> other = (Adapt<?>) obj;
        if (!resource.equals(other.resource))
            return false;
        if (allowNull != other.allowNull)
            return false;
        if (!target.equals(other.target))
            return false;
        if (uniqueResult != other.uniqueResult)
            return false;
        return true;
    }

}