/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.processor;

import java.util.concurrent.ConcurrentLinkedQueue;

import org.simantics.db.AsyncRequestProcessor;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.common.request.WriteOnlyRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.WriteOnly;

final public class MergingWriteOnlyProcessor extends ProcessorBase {

    final long transactionKeepalivePeriod;
    final ConcurrentLinkedQueue<WriteOnly> queue = new ConcurrentLinkedQueue<WriteOnly>(); 
    final private AsyncRequestProcessor processor;
    final private VirtualGraph vg;

    boolean hasAlreadyRequest = false;

    public MergingWriteOnlyProcessor(AsyncRequestProcessor processor, VirtualGraph vg, long transactionKeepalivePeriod) {
        this.processor = processor;
        this.vg = vg;
        this.transactionKeepalivePeriod = transactionKeepalivePeriod;
    }

    class MergedWrite extends WriteOnlyRequest {

    	public MergedWrite(VirtualGraph vg) {
    		super(vg);
		}
    	
        @Override
        public void perform(WriteOnlyGraph graph) throws DatabaseException {

//            System.out.println("MergedWrite begins");

            while (true) {
                
                WriteOnly next = queue.poll();
                if(next == null) {
                    synchronized (MergingWriteOnlyProcessor.this) {
                        if (transactionKeepalivePeriod > 0) {
                            try {
                                MergingWriteOnlyProcessor.this.wait(transactionKeepalivePeriod);
                            } catch (InterruptedException e) {
                	    		Logger.defaultLogError(e);
                            }
                            next = queue.poll();
                        }
                        if(next == null) {
                            hasAlreadyRequest = false;
//                            System.out.println("MergedWrite ends");
                            return;
                        }
                    }
                }

//                System.out.println("MergedWrite executes " + next);
                try {
                	next.perform(graph);
                } catch(Throwable t) {
    	    		Logger.defaultLogError(t);
                }

            }

        }

    }

    @Override
    public void asyncRequest(WriteOnly request) {
        
        queue.add(request);

        synchronized (this) {
        
          if (!hasAlreadyRequest) {
              processor.asyncRequest(new MergedWrite(vg));
              hasAlreadyRequest = true;
          } else {
              notify();
          }
          
        }

    }

    @Override
    public String toString() {
        return "MergingWriteOnlyProcessor@" + System.identityHashCode(this) + " (based on " + processor + ")";
    }

}
