package org.simantics.databoard.serialization.impl;

import gnu.trove.map.hash.TObjectIntHashMap;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.reflection.ClassBinding;
import org.simantics.databoard.binding.util.IsReferableQuery;
import org.simantics.databoard.binding.util.Result;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.serialization.Serializer.CompositeSerializer;

public class GenericRecordSerializer extends CompositeSerializer {

	RecordBinding binding;
	public Serializer[] componentSerializers;
	Integer fixedSize;
	int minSize;
	boolean createPartial;
	
	/**
	 * Field type for each field
	 *  0 - Object
	 *  1 - boolean
	 *  2 - byte
	 *  3 - int
	 *  4 - long
	 *  5 - float
	 *  6 - double
	 */
	public int[] fieldType;
	
	/**
	 * 
	 * @param binding
	 * @param componentSerializers (optional) can be set later
	 */
	public GenericRecordSerializer(RecordBinding binding, Serializer[] componentSerializers) {		
		super( IsReferableQuery.isReferable( binding.type() ) != Result.No );
		this.binding = binding;			
		this.componentSerializers = componentSerializers;
	}
	
	@Override
	public void finalizeConstruction() {
		fixedSize = null;

		createPartial = !binding.isImmutable();
		
		if ( binding instanceof ClassBinding ) {
			ClassBinding cb = (ClassBinding) binding;
			createPartial &= cb.ci.partialConstructionPossible;
		}
		
		fieldType = new int[componentSerializers.length];
		for (int i=0; i<componentSerializers.length; i++) {
			Serializer cs = componentSerializers[i];
			//createPartial &= !binding.getComponentBinding(i).isImmutable(); 
			minSize += cs.getMinSize();
			Integer componentFixedSize = cs.getConstantSize();
			if (componentFixedSize==null) {
				fixedSize = null;
				break;
			}
			fixedSize = fixedSize==null ? componentFixedSize : fixedSize+componentFixedSize;
			
			int type = 0;
			if ( cs instanceof BooleanSerializer ) type = 1;
			else if ( cs instanceof ByteSerializer ) type = 2;
			else if ( cs instanceof IntSerializer ) type = 3;
			else if ( cs instanceof LongSerializer ) type = 4;
			else if ( cs instanceof FloatSerializer ) type = 5;
			else if ( cs instanceof DoubleSerializer ) type = 6;
			fieldType[i] = type;
		}
		
//		if ( binding instanceof ClassBinding ) {
//			ClassBinding cb = (ClassBinding) binding;
//			System.out.println(cb.ci.clazz.getName()+" = "+createPartial);
//		}
		
	}

	@Override
	public Object deserialize(DataInput in, List<Object> identities) throws IOException {
//		assertRemainingBytes(in, minSize);
		try {
			if (createPartial) {
				Object obj = binding.createPartial();
				deserializeTo(in, identities, obj);
				return obj;
			} else {				
				Object[] temp = new Object[componentSerializers.length];
				for(int i=0;i<componentSerializers.length;++i) {
					temp[i] = componentSerializers[i].deserialize(in, identities);
				}
				return binding.create(temp);
			}
			
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public void deserializeTo(DataInput in, List<Object> identities, Object obj) throws IOException {
//		assertRemainingBytes(in, minSize);
		try {
			for(int i=0;i<componentSerializers.length;++i) {
				Serializer cs = componentSerializers[i];
				int type = fieldType[i];
				
				switch (type) {
				case 0:
					Binding cb = binding.componentBindings[i];
					boolean csImmutable = cb.isImmutable();
					if (csImmutable) {
						Object component = cs.deserialize(in, identities);
						binding.setComponent(obj, i, component);
					} else {
						Object component = binding.getComponent(obj, i);
						if ( component == null ) {
							component = cs.deserialize(in, identities);
							binding.setComponent(obj, i, component);
						} else {
							Object newComponent = cs.deserializeToTry(in, identities, component);
							if (newComponent!=component) binding.setComponent(obj, i, newComponent);
						}
					}
					break;
				case 1:
					binding.setBoolean(obj, i, ((BooleanSerializer)cs).getBoolean(in));
					break;
				case 2:
					binding.setByte(obj, i, ((ByteSerializer)cs).getByte(in));
					break;
				case 3:
					binding.setInt(obj, i, ((IntSerializer)cs).getInt(in));
					break;
				case 4:
					binding.setLong(obj, i, ((LongSerializer)cs).getLong(in));
					break;
				case 5:
					binding.setFloat(obj, i, ((FloatSerializer)cs).getFloat(in));
					break;
				case 6:
					binding.setDouble(obj, i, ((DoubleSerializer)cs).getDouble(in));
					break;
				}
				
			}
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}

	@Override
	public void skip(DataInput in, List<Object> identities) throws IOException, SerializationException {
		if (fixedSize != null) {
			in.skipBytes(fixedSize);
		} else {
			for(int i=0;i<componentSerializers.length;++i)
				componentSerializers[i].skip(in, identities);				
		}
	}		
	
	@Override
	public void serialize(DataOutput out, TObjectIntHashMap<Object> identities, Object obj) throws IOException {
		try {
			for(int i=0;i<componentSerializers.length;++i) {
				int type = fieldType[i];
				Serializer cs = componentSerializers[i];
				switch( type ) {
				case 0: cs.serialize(out, identities, binding.getComponent(obj, i));
					break;	
				case 1: ((BooleanSerializer)cs).putBoolean(out, binding.getBoolean(obj, i));
				break;
				case 2: ((ByteSerializer)cs).putByte(out, binding.getByte(obj, i));
				break;
				case 3: ((IntSerializer)cs).putInt(out, binding.getInt(obj, i));
				break;
				case 4: ((LongSerializer)cs).putLong(out, binding.getLong(obj, i));
				break;
				case 5: ((FloatSerializer)cs).putFloat(out, binding.getFloat(obj, i));
				break;
				case 6: ((DoubleSerializer)cs).putDouble(out, binding.getDouble(obj, i));
				break;
				}
			}
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}

	@Override
	public Integer getConstantSize() {			
		return fixedSize;
	}

	@Override
	public int getSize(Object obj, TObjectIntHashMap<Object> identities) throws IOException {
		try {
			if (fixedSize!=null) return fixedSize;
			int result = 0;
			for(int i=0;i<componentSerializers.length;++i)
				result += componentSerializers[i].getSize( binding.getComponent(obj, i), identities );			
			return result;
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	/**
	 * @return the minSize
	 */
	public int getMinSize() {
		return minSize;
	}
	
}
