/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.lang.reflect.Array;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.IntegerBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.util.IdentityPair;

/**
 * Binds Integer[] Type to int[] Class.
 */
public final class IntArrayBinding extends ArrayBinding {
	
	public static IntArrayBinding createFrom(ArrayType type) {
		return new IntArrayBinding(type, new IntegerBindingDefault((IntegerType) type.componentType));
	}

	public IntArrayBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
	}

	@Override
	public Object create() {
		return new int[0];
	}
	
	@Override
	public Object create(int length, Iterator<Object> values) {
		int[] array = new int[length];
		for(int i=0;i<length;++i)
			array[i] = (Integer)values.next();
		return array;
	}
	
	public Object create(Object[] values) {
		int[] result = new int[values.length];
		for (int i=0; i<values.length; i++)
			result[i] = (Integer) values[i];
		return result;
	}	
	
	/**
	 * Create an array object.
	 * Note! The argument is consumed (is used in the result).
	 * 
	 * @param array
	 * @return an object that contains the array
	 */
	public Object create(int[] array) {
		return array;
	}
	
	@Override
    public Object createDefault()
    throws BindingException
    {
		NumberType nt = (NumberType) type().componentType;
		if (nt.getRange() == null) {
	    	return new int[ type().minLength() ];
		}
		return super.createDefault();
    }	
	

	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		IntegerBinding scb = (IntegerBinding) sb.getComponentBinding();
		
		int[] d = (int[]) dst;
		if (d.length != sb.size(src)) throw new BindingException("int[] is length immutable");
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
	}	
	

	@Override
	public Object readFromTry(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		IntegerBinding scb = (IntegerBinding) sb.getComponentBinding();
		
		int[] d = (int[]) dst;
		int srcSize = sb.size(src);
		if (d.length != srcSize) d = new int[ srcSize ];
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
		return d;
	}		
	
	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", int[] expected");
		return ((int[])array)[index];
	}
	
	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		int[] a = (int[])array;
		for (int i=0; i<a.length; i++)
			result[i] = a[i];
	}	
	
	@Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		int[] a = (int[])array;
		a[index] = (Integer) value;
	}	

	@Override
	public int size(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", int[] expected");
		return ((int[])array).length;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof int[];
	}
	
	@Override
	public boolean isImmutable() {
		return false;
	}	
	
	public int[] getArray(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", int[] expected");
		return (int[]) array;
	}	

    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int result = 1;
    	int[] array = (int[]) value;
    	for (int i=0; i<array.length; i++) {
    		result = 31*result + array[i];
    	}
    	return result;
    }
	
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	int[] i1 = (int[]) o1;
    	int[] i2 = (int[]) o2;
		// Compare Lengths
		int l1 = i1.length;
		int l2 = i2.length;
		int dif = l1 - l2;
		if (dif!=0) return dif;
		// Compare elements
		for (int i=0; i<l1; i++) {
			int v1 = i1[i];
			int v2 = i2[i];
			dif = (v1<v2 ? -1 : (v1==v2 ? 0 : 1));
			if (dif!=0) return dif;
		}
		return 0;
    }    

    @Override
    public void add(Object array, int index, Object element)
    		throws BindingException, IndexOutOfBoundsException {
    	throw new UnsupportedOperationException();
    }
    
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
    	throw new UnsupportedOperationException();
	}

	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int oldSize = Array.getLength(array);
		if (oldSize==newSize) return;
		throw new BindingException("int[] is length immutable");
	}	

	@Override
	public boolean isResizable() {
		return false;
	}
    
}
