/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.factory;

import gnu.trove.map.hash.THashMap;
import gnu.trove.set.hash.THashSet;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.databoard.Datatypes;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.impl.HashMapBinding;
import org.simantics.databoard.binding.impl.HashSetBinding;
import org.simantics.databoard.binding.reflection.BindingProvider;
import org.simantics.databoard.binding.reflection.BindingRequest;
import org.simantics.databoard.binding.reflection.ClassBindingFactory;
import org.simantics.databoard.type.MapType;

public class TroveBindingsProvider implements BindingProvider {

	@Override
	public Binding provideBinding(ClassBindingFactory factory, BindingRequest request)
			throws BindingConstructionException {
		
        if ( request.getClazz().equals(THashSet.class) ) {
            MapType type = new MapType();
            type.valueType = Datatypes.VOID;            
           	return new THashSetBinding(type, null);
        }
		
        if ( request.getClazz().equals(THashMap.class) ) {
            MapType type = new MapType();
           	return new THashMapBinding(type, null, null);
        }
		
		return null;
	}

	static class THashMapBinding extends HashMapBinding {
		
		public THashMapBinding(Binding keyBinding, Binding valueBinding) {
			super(keyBinding, valueBinding);
		}

		public THashMapBinding(MapType mapType, Binding keyBinding,
				Binding valueBinding) {
			super(mapType, keyBinding, valueBinding);
		}

		@Override
		public Object create() {		
			return new THashMap<Object, Object>();
		}
		
		@Override
		public Object create(Object[] keys, Object[] values) {
			if (keys.length!=values.length)
				throw new IllegalArgumentException("Equal length arrays expected");
			
			int len = keys.length;
			THashMap<Object, Object> result = new THashMap<Object, Object>(len);
			
			for (int i=0; i<len; i++) {
				Object key = keys[i];
				Object value = values[i];
				result.put(key, value);
			}
			
			return result;
		}
		
		@Override
		public Object create(List<Object> keys, List<Object> values) {
			if (keys.size()!=values.size())
				throw new IllegalArgumentException("Equal length arrays expected");
			
			int len = keys.size();
			THashMap<Object, Object> result = new THashMap<Object, Object>(len);
			
			for (int i=0; i<len; i++) {
				Object key = keys.get(i);
				Object value = values.get(i);
				result.put(key, value);
			}
			
			return result;
		}
		
		@Override
		public Object create(Map<?, ?> initialMap) throws BindingException {
		    if (initialMap instanceof THashMap)
		        return initialMap;
		    
			// Replace with TreeMap. Create comparator from binding.
			THashMap<Object, Object> result = new THashMap<Object, Object>();
			putAll(result, initialMap);
			return result;
		}
		
		@Override
		public boolean isInstance(Object obj) {
			return obj instanceof THashMap;
		}
		
	}
	
	static class THashSetBinding extends HashSetBinding {
		
		public THashSetBinding(Binding elementBinding) {
			super(elementBinding);
			// TODO Auto-generated constructor stub
		}

		public THashSetBinding(MapType mapType, Binding elementBinding) {
			super(mapType, elementBinding);
			// TODO Auto-generated constructor stub
		}

		@Override
		public Object create() throws BindingException {
			return new THashSet<Object>();
		}

		public Object create(Set<?> initialSet) throws BindingException {
		    if (initialSet instanceof THashSet)
		        return initialSet;
		    
			return new THashSet<Object>(initialSet);
		}

		@SuppressWarnings({ "unchecked", "rawtypes" })
        @Override
		public Object create(Map initialMap) throws BindingException {
			return new THashSet<Object>(initialMap.keySet());
		}

		@Override
		public Object create(Object[] keys, Object[] values)
		throws BindingException {		
			THashSet<Object> result = new THashSet<Object>(keys.length);
			for (int i=0; i<keys.length; i++)
				result.add(keys[i]);
			return result;
		}
		
		@Override
		public Object create(List<Object> keys, List<Object> values) {
			THashSet<Object> result = new THashSet<Object>(keys.size());
			for (int i=0; i<keys.size(); i++)
				result.add(keys.get(i));
			return result;
		}
		
		@Override
		public boolean isInstance(Object obj) {
			return obj instanceof THashSet;
		}		
		
	}
	
}
