/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.IdentityHashMap;
import java.util.Set;

import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.impl.IntegerBindingDefault;
import org.simantics.databoard.binding.mutable.MutableIntegerBinding;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.util.IdentityPair;



/**
 * This is a binding of an Integer datatype. It is abstract and therfore
 * doesn't bind to any specific java class. Sub-classes bind to specific
 * classes.
 * 
 * Subclass IntegerJavaBinding binds to java.lang.Integer, and
 * MutableIntegerBinding to MutableInteger. 
 * 
 * @see IntegerType
 * @see IntegerBindingDefault java.lang.Integer binding
 * @see MutableIntegerBinding MutableInteger binding
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public abstract class IntegerBinding extends NumberBinding {
	
	public IntegerBinding(IntegerType type) {
		this.type = type;
	}
	
	@Override
	public IntegerType type() {
		return (IntegerType) type;
	}
	
	public abstract Object create(int value) throws BindingException;
	public abstract Object create(Integer value) throws BindingException;
	public abstract Object create(Number value) throws BindingException;
	public abstract Object create(String value) throws BindingException;
	public abstract Integer getValue(Object obj) throws BindingException;
	public abstract int getValue_(Object obj) throws BindingException;
	public abstract void setValue(Object obj, Number value) throws BindingException;
	public abstract void setValue(Object obj, int value) throws BindingException;
	public abstract boolean isInstance(Object obj);
	
    @Override
    public void accept(Visitor1 v, Object obj) {
        v.visit(this, obj);        
    }
    
    @Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }
    
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
   	
    	int v1 = getValue_(o1);
    	int v2 = getValue_(o2);
    	return (v1<v2 ? -1 : (v1==v2 ? 0 : 1));
    }
    
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int v = getValue_(value);
    	return v;
    }

	public Object createUnchecked(int value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	public Object createUnchecked(Integer value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
    
}

