/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.IdentityHashMap;
import java.util.Set;

import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.util.IdentityPair;


/**
 * This is a binding of Double Type and a Java Object.
 *
 * @see DoubleType
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public abstract class DoubleBinding extends NumberBinding {

	public DoubleBinding(DoubleType type) {
		this.type = type;
	}
	
	public DoubleType type() {
		return (DoubleType) type;
	}
	
	public abstract Object create(Double value) throws BindingException;
	public abstract Object create(double value) throws BindingException;
	public abstract Object create(Number value) throws BindingException;
	public abstract Object create(String value) throws BindingException;
	public abstract Double getValue(Object o) throws BindingException;
	public abstract double getValue_(Object o) throws BindingException;
	public abstract void setValue(Object obj, Number value) throws BindingException;
	public abstract void setValue(Object obj, double value) throws BindingException;
	public abstract boolean isInstance(Object obj);
    
    @Override
    public void accept(Visitor1 v, Object obj) {
        v.visit(this, obj);        
    }
    
    @Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }    
	
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	double v1 = getValue_(o1);
    	double v2 = getValue_(o2);
    	return Double.compare(v1, v2);    	
    }
    
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	double v = getValue_(value);
    	long bits = Double.doubleToLongBits(v);
    	return (int)(bits ^ (bits >>> 32));
    }
    
	public Object createUnchecked(double value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	public Object createUnchecked(Double value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	
}
