/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor;

import java.util.Map;

import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.accessor.impl.MapAccessorIterator;
import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.MapType;

/**
 * Map accessor is an interface to value container of map type
 *
 * @see MapAccessorIterator iterator helper class
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public interface MapAccessor extends Accessor {

	/**
	 * Get the number of elements in the map
	 * @return size
	 * @throws AccessorException
	 */
	int size() throws AccessorException;
	
	/**
	 * Get the value that is specified for a key 
	 * 
	 * @param keyBinding
	 * @param key
	 * @param valueBinding
	 * @return the value or <code>null</code>
	 * @throws AccessorException
	 */
	Object get(Binding keyBinding, Object key, Binding valueBinding) throws AccessorException;
	
	/**
	 * Returns true if map has a value for specified key
	 * 
	 * @param keyBinding
	 * @param key
	 * @return true if entry exists
	 * @throws AccessorException
	 */
	boolean containsKey(Binding keyBinding, Object key) throws AccessorException;
	
	/**
	 * Returns true if map has a value for one or multiple keys
	 * 
	 * @param valueBinding
	 * @param value
	 * @return true if value exists
	 * @throws AccessorException
	 */
	boolean containsValue(Binding valueBinding, Object value) throws AccessorException;
	
	/**
	 * Set & copy all entries from a Map Value.
	 *  
	 * If entries are removed and they have accessors, the accessors are invalidated. 
	 * 
	 * @param mapBinding
	 * @param newMap
	 * @throws BindingException binding error
	 * @throws UnsupportedOperationException cannot set a new value
	 */
	void setValue(Binding mapBinding, Object newMap) throws AccessorException;
	
	
	/**
	 * Put an entry to the map. If previous entry exists for the key, it is removed. 
	 * 
	 * @param keyBinding
	 * @param key
	 * @param valueBinding
	 * @param value
	 * @throws AccessorException
	 */
	void put(Binding keyBinding, Object key, Binding valueBinding, Object value) throws AccessorException;
	
	/**
	 * Remove an entry.
	 * 
	 * If there is an accessor, it becomes invalid.
	 * 
	 * @param keyBinding
	 * @param key
	 * @throws AccessorException
	 */
	void remove(Binding keyBinding, Object key) throws AccessorException;
	
	/**
	 * Remove all values between an two keys. 
	 * 
	 * If there is an existing accessor, it becomes invalid.
	 * 
	 * @param keyBinding
	 * @param startKey start key, inclusive
	 * @param endKey end key, exclusive
	 * @throws AccessorException
	 */
//	void removeRange(Binding keyBinding, Object startKey, Object endKey) throws AccessorException;
	
	/**
	 * Clear all entries.
	 * 
	 * If there is an accessor to a removed container, it becomes invalid.
	 * 
	 * @throws AccessorException
	 */
	void clear() throws AccessorException;
	
	/**
	 * Put entries from a map. Replaces any possible existing entry. 
	 * 
	 * @param keyBinding
	 * @param valueBinding
	 * @param from
	 * @throws AccessorException
	 */
	void putAll(Binding keyBinding, Binding valueBinding, Map<Object, Object> from) throws AccessorException;

	/**
	 * Put entries from a map. Replaces any possible existing entry. 
	 * 
	 * @param keyBinding
	 * @param valueBinding
	 * @param keys
	 * @param values
	 * @throws AccessorException
	 */
	void putAll(Binding keyBinding, Binding valueBinding, Object[] keys, Object[] values) throws AccessorException;
	
	/**
	 * Get all entries, write to a Java Collection
	 * 
	 * @param keyBinding
	 * @param valueBinding
	 * @param to
	 * @throws AccessorException
	 */
	void getAll(Binding keyBinding, Binding valueBinding, Map<Object, Object> to) throws AccessorException;
	
	/**
	 * Get all entries in order, write to 2 arrays.  
	 * 
	 * @param keyBinding
	 * @param valueBinding
	 * @param keys
	 * @param values
	 * @throws AccessorException
	 */
	void getAll(Binding keyBinding, Binding valueBinding, Object[] keys, Object[] values) throws AccessorException;
	
	/**
	 * Count the number of entries between two keyes
	 * @param keyBinding
	 * @param from
     * @param fromInclusive
	 * @param end 
     * @param endInclusive
     * @return number of entries in range 
	 * @throws AccessorException
	 */
	int count(Binding keyBinding, Object from, boolean fromInclusive, Object end, boolean endInclusive) throws AccessorException;

	/**
	 * Read a range of entries
	 * 
	 * @param keyBinding
	 * @param from
	 * @param fromInclusive
	 * @param end
	 * @param endInclusive
	 * @param keyArrayBinding
	 * @param keysArray
	 * @param valueArrayBinding
	 * @param valueArray
	 * @param resultLimit maximum number of entries to read, -1 for no limit
	 * @return the number of entries read 
	 * @throws AccessorException
	 */
	int getEntries(Binding keyBinding, Object from, boolean fromInclusive, Object end, boolean endInclusive, 
			ArrayBinding keyArrayBinding, Object keysArray, 
			ArrayBinding valueArrayBinding, Object valueArray, 
			int resultLimit) throws AccessorException;	
	
	/**
	 * Get all keys in order
	 * 
	 * @param keyBinding
	 * @return an array or keys
	 * @throws AccessorException
	 */
	Object[] getKeys(Binding keyBinding) throws AccessorException;
	
	/**
	 * Get all values
	 * 
	 * @param valueBinding
	 * @return an array of values
	 * @throws AccessorException
	 */
	Object[] getValues(Binding valueBinding) throws AccessorException;
	
	/**
	 * Get an accessor to a value. It becomes invalid if the entry is
	 * removed or overwritten with a new value.
	 * 
	 * @param keyBinding
	 * @param key
	 * @return value accessor
	 * @throws AccessorConstructionException
	 */
	<T extends Accessor> T getValueAccessor(Binding keyBinding, Object key) throws AccessorConstructionException;
	
	MapType type();
 
	Object getFirstKey(Binding keyBinding) throws AccessorException;
	Object getLastKey(Binding keyBinding) throws AccessorException;
	Object getLowerKey(Binding keyBinding, Object key) throws AccessorException;
	Object getFloorKey(Binding keyBinding, Object key) throws AccessorException;
	Object getCeilingKey(Binding keyBinding, Object key) throws AccessorException;
	Object getHigherKey(Binding keyBinding, Object key) throws AccessorException;
	
}

