/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor;

import java.util.Collection;

import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.ArrayType;

public interface ArrayAccessor extends Accessor {

	/**
	 * Add a new value. 
	 * 
	 * @param binding
	 * @param value value
	 * @throws AccessorException
	 */
	void add(Binding binding, Object value) throws AccessorException;

	/**
	 * Add an array of elements.
	 * 
	 * @param binding
	 * @param values value 
	 * @throws AccessorException
	 */
	void addAll(Binding binding, Object[] values) throws AccessorException;
	
	/**
	 * Add an array of elements.  
	 * 
	 * If elements are inserted in the middle of the array, existing interest sets
	 * are updated to reflect the new positions. 
	 * 
	 * @param index position to insert new value to
	 * @param binding
	 * @param values
	 * @throws AccessorException
	 */
	void addAll(int index, Binding binding, Object[] values) throws AccessorException;
	
	/**
	 * Insert a new value.
	 * 
	 * If elements are inserted in the middle of the array, existing interest sets
	 * are updated to reflect the new positions. 
	 * 
	 * @param index position to insert new value to
	 * @param binding
	 * @param value value
	 * @throws AccessorException
	 */
	void add(int index, Binding binding, Object value) throws AccessorException;
	
	/**
	 * Set all elements from an Array Value.
	 * 
	 * If array becomes shorter and there are accessors to the removed elements,
	 * the accessors are invalidated. 
	 * 
	 * @param binding
	 * @param newValue
	 * @throws BindingException binding error
	 * @throws UnsupportedOperationException cannot set a new value
	 */
	void setValue(Binding binding, Object newValue) throws AccessorException;
	
	/**
	 * Replace a value container with a new value.
	 * 
	 * @param index
	 * @param binding
	 * @param value
	 * @throws AccessorException
	 */
	void set(int index, Binding binding, Object value) throws AccessorException;
	
	/**
	 * Remove an element at an index.
	 *  
	 * If there are listeners to elements after the <code>index</code>, the 
	 * interest sets and accessor paths are updated and decreased.    
	 * 
	 * If there was an accessor, it becomes invalid.
	 * 
	 * @param index
	 * @param count
	 * @throws AccessorException
	 */
	void remove(int index, int count) throws AccessorException;
	
	/**
	 * Get an accessor to an element. 
	 * The accessor is invalidated if the element is removed from the array.
	 * <p>
	 * The accessor is not to the index, it is to the element. For instance, you get accessor X of [2]
	 * then a new value is inserted before 2. The accessor now points to the element at [3].   
	 * 
	 * @param index
	 * @return the accessor
	 * @throws AccessorConstructionException
	 */
	<T extends Accessor> T getAccessor(int index) throws AccessorConstructionException;
	
	/**
	 * Get a copy of the element 
	 * 
	 * @param index
	 * @param valueBinding
	 * @return the element
	 * @throws AccessorException
	 */
	Object get(int index, Binding valueBinding) throws AccessorException;
	
	/**
	 * Read the element to an object
	 * 
	 * @param index
	 * @param valueBinding
	 * @param dest
	 * @throws AccessorException
	 */
	void get(int index, Binding valueBinding, Object dest) throws AccessorException;
	
	/**
	 * Get all elements and place them to an <code>array</code>. 
	 * Exception is thrown if Array length is too short. 
	 * 
	 * @param valueBinding
	 * @param array 
	 * @throws AccessorException
	 */
	void getAll(Binding valueBinding, Object[] array) throws AccessorException;
	
	/**
	 * Get all elements
	 * 
	 * @param valueBinding
	 * @param values
	 * @throws AccessorException
	 */
	void getAll(Binding valueBinding, Collection<Object> values) throws AccessorException;
	
//	/**
//	 * Get a partial array 
//	 * 
//	 * @param binding
//	 * @param index
//	 * @param count
//	 * @return array
//	 * @throws AccessorException
//	 */
//	Object getPartial(ArrayBinding binding, int index, int count) throws AccessorException;
	
	void setSize(int newSize) throws AccessorException;
	
	/**
	 * Return the number of elements in the array.
	 * 
	 * @return the number of elements
	 * @throws AccessorException
	 */
	int size() throws AccessorException;	
	
	ArrayType type();

	public interface CloseableArrayAccessor extends ArrayAccessor, CloseableAccessor {}
	
}


