/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.util.binary;

import java.io.DataInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class ByteBufferReadable implements BinaryReadable {
	
	ByteBuffer buf;
	
	public ByteBufferReadable(ByteBuffer buf) {		
		if (buf == null)
			throw new IllegalArgumentException("null");
		this.buf = buf;
	}

	public ByteBufferReadable(byte[] buf) {		
		if (buf == null)
			throw new IllegalArgumentException("null");
		this.buf = ByteBuffer.wrap(buf);		
	}

	
	@Override
	public byte readByte() {
		return buf.get();
	}
		
	int _read() {
		if (buf.position()>=buf.limit()) return -1;
		return buf.get() & 0xff;
	}
	
    public final String readLine() throws IOException {
    	StringBuffer input = new StringBuffer();
    	int c = -1;
    	boolean eol = false;

    	while (!eol) {
    	    switch (c = _read()) {
    	    case -1:
    	    case '\n':
    		eol = true;
    		break;
    	    case '\r':
    		eol = true;
    		long cur = position();
    		if ((_read()) != '\n') {
    		    position(cur);
    		}
    		break;
    	    default:
    		input.append((char)c);
    		break;
    	    }
    	}

    	if ((c == -1) && (input.length() == 0)) {
    	    return null;
    	}
    	return input.toString();
    }	
	
	
	@Override
	public int readUnsignedByte() throws IOException {
		return buf.get() & 0x000000ff;
	}		
	
	@Override
	public boolean readBoolean() throws IOException {
		return buf.get()!=0;
	}

	@Override
	public void readFully(byte[] dst, int offset, int length) {
		buf.get(dst, offset, length);
	}

	@Override
	public void readFully(byte[] dst) {
		buf.get(dst);
	}

	@Override
	public void readFully(ByteBuffer buf) {		
		if (buf.hasArray()) {
			this.buf.get(buf.array(), buf.arrayOffset() + buf.position(), buf.remaining());
			buf.position(buf.capacity());
		} else {
			buf.put(buf);
		}
	}

	@Override
	public void readFully(ByteBuffer buf, int length) {
		if (buf.hasArray()) {
			this.buf.get(buf.array(), buf.arrayOffset() + buf.position(), length);
			buf.position(buf.position() + length);
		} else {
//			int len = Math.min( Math.min( buf.remaining(), this.buf.remaining() ), length);
			int len = length;
			int origLimit = this.buf.limit();
			try {
				this.buf.limit(this.buf.position()+len);
				buf.put(this.buf);
			} finally {
				this.buf.limit(origLimit);
			}
		}
	}

	@Override
	public double readDouble() {
		return buf.getDouble();
	}

	@Override
	public float readFloat() {
		return buf.getFloat();
	}

	@Override
	public int readInt() {
		return buf.getInt();
	}

	@Override
	public long readLong() {
		return buf.getLong();
	}

	@Override
	public short readShort() {
		return buf.getShort();
	}
	
    public final String readUTF() throws IOException {
    	return DataInputStream.readUTF(this);
    } 
    
	@Override
	public char readChar() throws IOException {
		return buf.getChar();
	}

	@Override
	public int readUnsignedShort() {
		return buf.getShort() & 0xffff;
	}
	
	@Override
	public long length() {
		return buf.limit();
	}
	
	@Override
	public long position() {
		return buf.position();
	}
	
	public ByteOrder order() {
		return buf.order();
	}

	public void order(ByteOrder order) {
		buf.order(order);
	}

	public void position(int newPosition) throws IOException {
		buf.position(newPosition);
	}

	public void position(long newPosition) throws IOException {
		if (newPosition>=Integer.MAX_VALUE || newPosition<0) throw new IllegalArgumentException("index out of range");
		buf.position((int) newPosition);
	}

	@Override
	public long skipBytes(long bytes) throws IOException {
		long newPosition = bytes + position();
		position( newPosition );
		return bytes;
	}

	@Override
	public int skipBytes(int bytes) throws IOException {
		long newPosition = bytes + position();
		position( newPosition );
		return bytes;
	}
	
}
