package org.simantics.databoard.serialization.impl;

import gnu.trove.map.hash.TObjectIntHashMap;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.serialization.Serializer.RecursiveSerializer;
import org.simantics.databoard.type.RecordType;

public class ReferableRecordSerializer extends RecursiveSerializer {

	RecordBinding binding;
	public Serializer[] componentSerializers;
	
	/** The combined size elements, or <code>null</code> */ 
	Integer fixedSizeOfContent;
	int minContentSize;
	
	public ReferableRecordSerializer(RecordBinding binding, Serializer[] componentSerializers) {
		this.binding = binding;
		this.componentSerializers = componentSerializers;
	}
	
	@Override
	public void finalizeConstruction() {
		fixedSizeOfContent = null;
		for (Serializer componentSerializer : componentSerializers) {
			minContentSize += componentSerializer.getMinSize();
			Integer componentFixedSize = componentSerializer.getConstantSize();
			if (componentFixedSize==null) {
				fixedSizeOfContent = null;
				break;
			}
			fixedSizeOfContent = fixedSizeOfContent==null ? componentFixedSize : fixedSizeOfContent + componentFixedSize; 
		}
	}

	boolean referable() {
		return ((RecordType)binding.type()).isReferable();
	}
	
	@Override
	public Object deserialize(DataInput in, List<Object> identities) throws IOException {
		try {
			int id = in.readInt();
			if(id > 0)
				return identities.get(id-1);			
			
			Object result = binding.createPartial();
			identities.add(result);
			
//			assertRemainingBytes(in, minContentSize);
			Object[] temp = new Object[componentSerializers.length];
			for(int i=0;i<componentSerializers.length;++i)
				temp[i] = componentSerializers[i].deserialize(in, identities);
			binding.setComponents(result, temp);
			
			return result;
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public void deserializeTo(DataInput in, List<Object> identities,
			Object obj) throws IOException {
		try {
			int id = in.readInt();
			if(id > 0) {
				Object alreadyDeserializedObject = identities.get(id-1);
				if (alreadyDeserializedObject!=obj) {
					for (int i=0; i<binding.getComponentCount(); ++i) {
						Object component = binding.getComponent(alreadyDeserializedObject, i);
						binding.setComponent(obj, i, component);
					}
				}
				return;
			}
			
			identities.add(obj);
//			assertRemainingBytes(in, minContentSize);
			for(int i=0;i<componentSerializers.length;++i) {
				Serializer cs = componentSerializers[i];
				boolean csImmutable = binding.getComponentBinding(i).isImmutable();
				if (csImmutable) {
					Object component = cs.deserialize(in, identities);
					binding.setComponent(obj, i, component);
				} else {
					Object component = binding.getComponent(obj, i);
					component = cs.deserializeToTry(in, identities, component);
					binding.setComponent(obj, i, component);
				}
			}
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
		
	}
	
	@Override
	public void skip(DataInput in, List<Object> identities) throws IOException, SerializationException {
		
		int id = in.readInt();
		if(id > 0) return;			
		
		for(int i=0;i<componentSerializers.length;++i)
			componentSerializers[i].skip(in, identities);
	}

	@Override
	public void serialize(DataOutput out, TObjectIntHashMap<Object> identities, Object obj) throws IOException {
		try {
			int id = identities.get(obj);
			if(id > 0) {
				out.writeInt(id);
			}
			else {
				out.writeInt(0);
				identities.put(obj, identities.size() + 1);
				for(int i=0;i<componentSerializers.length;++i)
					componentSerializers[i].serialize(out, identities, binding.getComponent(obj, i));
			}
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
			
	}

	@Override
	public Integer getConstantSize() {
		return null;
	}

	@Override
	public int getSize(Object obj, TObjectIntHashMap<Object> identities) throws IOException {
		try {
			int id = identities.get(obj);
			if (id>0) return 4;
			identities.put(obj, identities.size()+1);
			if (fixedSizeOfContent!=null) return 4+fixedSizeOfContent;
			int result = 4;			
			for(int i=0;i<componentSerializers.length;++i)
				result += componentSerializers[i].getSize( binding.getComponent(obj, i), identities );			
			return result;
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
			
	}
	
	@Override
	public int getMinSize() {
		return 4;
	}
	
}

