package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.InputStream;

import org.simantics.databoard.binding.StringBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.util.binary.BinaryWriteable;
import org.simantics.databoard.util.binary.Endian;
import org.simantics.databoard.util.binary.UTF8;

/**
 * Serializes strings to Modified UTF-8, 
 * see http://download.oracle.com/javase/6/docs/api/java/io/DataInput.html
 * 
 * @author Toni Kalajainen
 */
public class ModifiedUTF8StringSerializer extends NonRecursiveSerializer {

	StringBinding binding;

	public ModifiedUTF8StringSerializer(StringBinding binding) {
		this.binding = binding;
	}

	@Override
	public Object deserialize(byte[] data) throws IOException {
		try {
			int dataLength = data.length;
			if (dataLength == 0)
				throw new SerializationException("Malformed data. Deserialization aborted. No string length data.");
			long l = Endian.readDynamicUInt32(data, 0);
			int stringLength = (int) (l & 0xFFFF_FFFFL);
			int lengthBytes = (int) ((l>>>32) & 0xFFFF_FFFFL);
			if (dataLength < lengthBytes + stringLength)
				throw new SerializationException("Malformed data. Deserialization aborted. (Wrong binding?)");
			String str = UTF8.readModifiedUTF(data, lengthBytes, dataLength);
			return binding.create(str);
		} catch (BindingException e) {
			throw new IOException(e);
		}
	}

	@Override
	public void deserialize(byte[] data, Object obj) throws IOException {
		try {
			int dataLength = data.length;
			if (dataLength == 0)
				throw new SerializationException("Malformed data. Deserialization aborted. No string length data.");
			long l = Endian.readDynamicUInt32(data, 0);
			int stringLength = (int) (l & 0xFFFF_FFFFL);
			int lengthBytes = (int) ((l>>>32) & 0xFFFF_FFFFL);
			if (dataLength < lengthBytes + stringLength)
				throw new SerializationException("Malformed data. Deserialization aborted. (Wrong binding?)");
			String str = UTF8.readModifiedUTF(data, lengthBytes, data.length);
			binding.setValue(obj, str);
		} catch (BindingException e) {
			throw new IOException( e );
		}
	}

	@Override
	public Object deserialize(DataInput in) throws IOException {
		try {
			int len = Endian.readDynamicUInt32(in);
			assertRemainingBytes(in, len); 
			String str = UTF8.readModifiedUTF(in, len);
			return binding.create(str);
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public void deserializeTo(DataInput in, Object obj) throws IOException {
		try {
			int len = Endian.readDynamicUInt32(in);
			assertRemainingBytes(in, len); 
			String str = UTF8.readModifiedUTF(in, len);
			binding.setValue(obj, str);
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public void skip(DataInput in) throws IOException, SerializationException {
		int length = Endian.readDynamicUInt32(in);
		in.skipBytes(length);
	}
	
	@Override
	public void transfer(InputStream in, BinaryWriteable w) throws IOException {
		int length = Endian.readDynamicUInt32(in);
		Endian.writeDynamicUInt32(w, length);
		for(int i=0;i<length;i++) {
			w.write(in.read());
		}
	}
	
	@Override
	public void serialize(DataOutput out, Object obj) throws IOException {
		try {
			String str = binding.getValue(obj);
			int utflen = UTF8.getModifiedUTF8EncodingByteLength(str);
			Endian.writeDynamicUInt32(out, utflen);
			UTF8.writeModifiedUTF(out, str);
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public Integer getConstantSize() {
		return null;
	}

	@Override
	public int getSize(Object obj) throws IOException {
		try {
			String string = binding.getValue(obj);
			int stringByteLength = UTF8.getModifiedUTF8EncodingByteLength( string ); 
			int lengthLength = Endian.getDynamicUInt32Length( stringByteLength );
			return lengthLength + stringByteLength;
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public int getMinSize() {
		return 1;
	}

}