/*******************************************************************************
 *  Copyright (c) 2025 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.InputStream;

import org.simantics.databoard.binding.StringBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.serialization.Sized;
import org.simantics.databoard.util.binary.BinaryWriteable;
import org.simantics.databoard.util.binary.Endian;
import org.simantics.databoard.util.binary.UTF8;

/**
 * Serializes strings to Modified UTF-8 
 * see http://download.oracle.com/javase/6/docs/api/java/io/DataInput.html
 * 
 * But with special handling of length. Uses a constant 4-byte length header which
 * is used only when size cannot be implied from given buffer/stream 
 * 
 * @author Antti Villberg
 */
public class AppendableModifiedUTF8StringSerializer extends NonRecursiveSerializer {

	StringBinding binding;

	public AppendableModifiedUTF8StringSerializer(StringBinding binding) {
		this.binding = binding;
	}

	@Override
	public Object deserialize(byte[] data) throws IOException {
		try {
			// Use size of given buffer, just ignore the length in header
			return binding.create(UTF8.readModifiedUTF(data, 4, data.length));
		} catch (BindingException e) {
			throw new IOException(e);
		}
	}

	@Override
	public void deserialize(byte[] data, Object obj) throws IOException {
		try {
			// Use size of given buffer, just ignore the length in header
			binding.setValue(obj, UTF8.readModifiedUTF(data, 4, data.length));
		} catch (BindingException e) {
			throw new IOException( e );
		}
	}

	@Override
	public Object deserialize(DataInput in) throws IOException {
		try {
			// Actually use the size from header
			int len = in.readInt();
			assertRemainingBytes(in, len); 
			return binding.create(UTF8.readModifiedUTF(in, len));
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public void deserializeTo(DataInput in, Object obj) throws IOException {
		try {
			// Actually use the size from header
			int len = in.readInt();
			assertRemainingBytes(in, len); 
			binding.setValue(obj, UTF8.readModifiedUTF(in, len));
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public void skip(DataInput in) throws IOException, SerializationException {
		// Actually use the size from header
		int len = in.readInt();
		assertRemainingBytes(in, len); 
		in.skipBytes(len);
	}
	
	
	public void transfer(InputStream in, BinaryWriteable w) throws IOException
	{
		int length = Endian.readInt(in);
		if(in instanceof Sized) {
			length = ((Sized) in).getAmountOfBytes() - 4;
		}
		w.writeInt(length);
		for(int i=0;i<length;i++) {
			w.write(in.read());
		}
	}
	
	@Override
	public void serialize(DataOutput out, Object obj) throws IOException {
		try {
			String str = binding.getValue(obj);
			out.writeInt(UTF8.getModifiedUTF8EncodingByteLength(str));
			UTF8.writeModifiedUTF(out, str);
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public Integer getConstantSize() {
		return null;
	}

	@Override
	public int getSize(Object obj) throws IOException {
		try {
			return 4 + UTF8.getModifiedUTF8EncodingByteLength( binding.getValue(obj) );
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	@Override
	public int getMinSize() {
		return 4;
	}

}