package org.simantics.databoard.serialization;

import java.util.Map;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.BooleanBinding;
import org.simantics.databoard.binding.ByteBinding;
import org.simantics.databoard.binding.DoubleBinding;
import org.simantics.databoard.binding.FloatBinding;
import org.simantics.databoard.binding.IntegerBinding;
import org.simantics.databoard.binding.LongBinding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.OptionalBinding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.StringBinding;
import org.simantics.databoard.binding.UnionBinding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.impl.BooleanArrayBinding;
import org.simantics.databoard.binding.impl.ByteArrayBinding;
import org.simantics.databoard.binding.impl.DoubleArrayBinding;
import org.simantics.databoard.binding.impl.FloatArrayBinding;
import org.simantics.databoard.binding.impl.IntArrayBinding;
import org.simantics.databoard.binding.impl.LongArrayBinding;
import org.simantics.databoard.serialization.impl.ArraySerializer;
import org.simantics.databoard.serialization.impl.BooleanArraySerializer;
import org.simantics.databoard.serialization.impl.BooleanSerializer;
import org.simantics.databoard.serialization.impl.ByteArraySerializer;
import org.simantics.databoard.serialization.impl.ByteSerializer;
import org.simantics.databoard.serialization.impl.DoubleArraySerializer;
import org.simantics.databoard.serialization.impl.DoubleSerializer;
import org.simantics.databoard.serialization.impl.FloatArraySerializer;
import org.simantics.databoard.serialization.impl.FloatSerializer;
import org.simantics.databoard.serialization.impl.GenericRecordSerializer;
import org.simantics.databoard.serialization.impl.IntArraySerializer;
import org.simantics.databoard.serialization.impl.IntSerializer;
import org.simantics.databoard.serialization.impl.LongArraySerializer;
import org.simantics.databoard.serialization.impl.LongSerializer;
import org.simantics.databoard.serialization.impl.MapSerializer;
import org.simantics.databoard.serialization.impl.ModifiedUTF8StringSerializer;
import org.simantics.databoard.serialization.impl.MutableVariantSerializer;
import org.simantics.databoard.serialization.impl.OptionalSerializer;
import org.simantics.databoard.serialization.impl.ReferableRecordSerializer;
import org.simantics.databoard.serialization.impl.UnionSerializer;
import org.simantics.databoard.serialization.impl.VariantSerializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;

/**
 * Default serializer factory creates serializers that follow databoard serialization
 * specification.
 * 
 * <a href="http://dev.simantics.org/index.php/Org.simantics.datatype_Manual#Binary_Serialization">Binary Serialization format</a>
 *
 * 
 * @author Toni Kalajainen
 */
public class DefaultSerializerFactory extends SerializerFactory {

	/**
	 * Construct a new serializer.
	 */
	public DefaultSerializerFactory() {
		super();
	}
	
	/**
	 * Construct a new serializer factory that places constructed serializers
	 * into user given repository.
	 * 
	 * @param repository
	 */
	public DefaultSerializerFactory(Map<Binding, Serializer> repository) {
		super(repository);
	}	

	@Override
	protected Serializer doConstruct(Binding binding)
			throws SerializerConstructionException {
	    
	    // Specialized serializers
	    if(binding instanceof SpecializedSerializerProvider) {
	        Serializer specializedSerializer = 
	            ((SpecializedSerializerProvider)binding).getSpecializedSerializer();
	        if(specializedSerializer != null)
	            return specializedSerializer;
	    }
		
		// Primitives
		if (binding instanceof BooleanBinding) return new BooleanSerializer( (BooleanBinding) binding );
		if (binding instanceof ByteBinding) return new ByteSerializer( (ByteBinding) binding );
		if (binding instanceof IntegerBinding) return new IntSerializer( (IntegerBinding) binding );
		if (binding instanceof LongBinding) return new LongSerializer( (LongBinding) binding );
		if (binding instanceof FloatBinding) return new FloatSerializer( (FloatBinding) binding );
		if (binding instanceof DoubleBinding) return new DoubleSerializer( (DoubleBinding) binding );
		if (binding instanceof StringBinding) return new ModifiedUTF8StringSerializer( (StringBinding) binding );

		// Record
		if (binding instanceof RecordBinding) {
			RecordBinding b = (RecordBinding) binding;
			Binding[] componentBindings = b.getComponentBindings();
			int count = b.getComponentCount();
			
			if ( b.type().isReferable() ) {
				ReferableRecordSerializer result = new ReferableRecordSerializer(b, null);
				inprogress.put(binding, result);				
				result.componentSerializers = new Serializer[count];
				for (int i=0; i<count; i++) {
					result.componentSerializers[i] = construct( componentBindings[i] );
				}				
				result.finalizeConstruction();
				inprogress.remove(binding);
				return result;
			} else {
				GenericRecordSerializer result = new GenericRecordSerializer(b, null);
				inprogress.put(binding, result);				
				result.componentSerializers = new Serializer[count];
				for (int i=0; i<count; i++) {
					result.componentSerializers[i] = construct( componentBindings[i] );
				}				
				result.finalizeConstruction();
				inprogress.remove(binding);
				return result;					
			}
		}		
		
		// Union
		if (binding instanceof UnionBinding) {
			UnionBinding b = (UnionBinding) binding;
			Binding[] componentBindings = b.getComponentBindings();
			int count = b.getComponentCount();
			
			UnionSerializer result = new UnionSerializer(b, null);
			inprogress.put(binding, result);				
			result.componentSerializers = new Serializer[count];
			for (int i=0; i<count; i++) {
				result.componentSerializers[i] = construct( componentBindings[i] );
			}				
			result.finalizeConstruction();
			inprogress.remove(binding);
			return result;
		}			
		
		// Optional
		if (binding instanceof OptionalBinding) {
			OptionalBinding b = (OptionalBinding) binding;
			OptionalSerializer result = new OptionalSerializer(b, null);
			inprogress.put(binding, result);
			result.componentSerializer = construct( b.getComponentBinding() );
			inprogress.remove(binding);
			return result;			
		}		
		
		// Array
		if (binding instanceof ArrayBinding) {
			ArrayBinding b = (ArrayBinding) binding;
			ArrayType type = (ArrayType) b.type();
			
			if( b instanceof FloatArrayBinding && type.componentType instanceof FloatType )
				return new FloatArraySerializer(b);
			if(b instanceof DoubleArrayBinding && type.componentType instanceof DoubleType )
				return new DoubleArraySerializer(b);
			if(b instanceof IntArrayBinding && type.componentType instanceof IntegerType )
				return new IntArraySerializer(b);
			if(b instanceof ByteArrayBinding && type.componentType instanceof ByteType )
				return new ByteArraySerializer(b);
			if(b instanceof BooleanArrayBinding && type.componentType instanceof BooleanType )
				return new BooleanArraySerializer(b);
			if(b instanceof LongArrayBinding && type.componentType instanceof LongType )
				return new LongArraySerializer(b);
			
			ArraySerializer result = new ArraySerializer(b, null);
			inprogress.put(binding, result);
			result.componentSerializer = construct( b.getComponentBinding() );
			result.finalizeConstruction();
			inprogress.remove(binding);
			return result;			
		}
		
		// Map
		if (binding instanceof MapBinding) {
			MapBinding b = (MapBinding) binding;			
			MapSerializer result = new MapSerializer(b, null, null);
			inprogress.put(binding, result);
			result.keySerializer = construct( b.getKeyBinding() );
			result.valueSerializer = construct( b.getValueBinding() );
			result.finalizeConstruction();
			inprogress.remove(binding);
			return result;			
		}		
		
		// Variant
		if (binding instanceof VariantBinding) {
			VariantSerializer result = binding.isImmutable() ? new VariantSerializer((VariantBinding)binding, this)
			                                                 : new MutableVariantSerializer((VariantBinding)binding, this );
			result.finalizeConstruction();
			return result;
		}
		
		throw new SerializerConstructionException("Cannot serialize "+binding.getClass().getName());
	}

	@Override
	public boolean supportsBinding(Binding binding) {
		return true;
	}
	
}
