package org.simantics.databoard.primitives;


/**
 * Unsigned 32-bit integer value, either mutable or immutable. 
 * The value is between 0 and 4294967295.
 * 
 * Example use: 
 *    UnsignedInteger x = new UnsignedInteger.Mutable(); 
 *    UnsignedInteger y = new UnsignedInteger.Immutable(4); 
 *    UnsignedInteger z = UnsingedInteger.valueOf(5);
 *
 * @author Toni Kalajainen <toni.kalajainen@iki.fi>
 */
public abstract class UnsignedInteger extends Number implements Comparable<Number> {

	private static final long serialVersionUID = 1L;

	public static final UnsignedInteger MIN_VALUE, MAX_VALUE, ZERO;
	
	static long MASK = 0xFFFFFFFFl;
	public static final long L_MAX_VALUE = 0xFFFFFFFFL;
	public static final long L_MIN_VALUE = 0;

	/** Value container */
	int value;
	
    public static UnsignedInteger fromBits(int intBits) {
		if (intBits>=0 && intBits<CACHE.length)	return CACHE[intBits];
		return UnsignedInteger.Immutable.fromBits(intBits);
    }	

    public static UnsignedInteger valueOf(long value) {
		if ( value>=0 && value<CACHE.length ) return CACHE[(int)value];
    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");		
		return new Immutable(value);    	
    }
	
	public static class Mutable extends UnsignedInteger {

		private static final long serialVersionUID = 1L;

		public Mutable() {}
		
		/**
		 * Create new unsigned integer from a signed integer.
		 * Use #fromBits() to make bitwise conversion
		 * 
		 * @param value signed integer
		 * @throws IllegalArgumentException if argument is sub zero
		 */
	    public Mutable(int value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = value;
	    }

	    public Mutable(long value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }

	    public Mutable(String stringValue) throws IllegalArgumentException {
	        long value = Long.parseLong(stringValue);
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	
	    
	    public static UnsignedInteger fromBits(int intBits) {
	    	Mutable result = new Mutable();
	    	result.value = intBits;
	    	return result;
	    }	
	    
	    public void setBits(int intBits) {
	    	this.value = intBits;
	    }
	    
	    public void setValue(int value) {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	    	this.value = value;
	    }
	    
	    public void setValue(long value) {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	    
		
	}
	
	public final static class Immutable extends UnsignedInteger {

		private static final long serialVersionUID = 1L;	

		Immutable() {}
		
		/**
		 * Create new unsigned integer from a signed integer.
		 * Use #fromBits() to make bitwise conversion
		 * 
		 * @param value signed integer
		 * @throws IllegalArgumentException if argument is sub zero
		 */
	    public Immutable(int value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = value;
	    }

	    public Immutable(long value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }

	    public Immutable(String stringValue) throws IllegalArgumentException {
	        long value = Long.parseLong(stringValue);
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	
	    			    
	}	
	    
    public int toBits() {
    	return value;
    }
	
	@Override
	public int intValue() {
		return value;
	}
	
	@Override
	public long longValue() {
		return value & MASK;
	}
	
	@Override
	public float floatValue() {
		return value & MASK;
	}
	@Override
	public double doubleValue() {
		return value & MASK;
	}
	
    @Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (obj == this) return true;
		
		if (obj instanceof UnsignedInteger == false) return false;
		UnsignedInteger other = (UnsignedInteger) obj;
		return value == other.value;
	}
    
    @Override
    public String toString() {
        return Long.toString(value & MASK);
    }
    
	@Override
	public int compareTo(Number obj) {
        return Long.signum( (value & MASK) - obj.longValue() );
	}
	
	@Override
	public int hashCode() {
		return value;
	}

	// Initialize Cache
	private static int CACHE_SIZE = 16;
	private static UnsignedInteger.Immutable[] CACHE;
	static {
		CACHE = new UnsignedInteger.Immutable[CACHE_SIZE];
		for (int i=0; i<CACHE_SIZE; i++) CACHE[i] = new UnsignedInteger.Immutable(i);
		ZERO = MIN_VALUE = CACHE[0];
		MAX_VALUE = new UnsignedInteger.Immutable(L_MAX_VALUE);
	}		    
	
}
