/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.method;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;

import org.simantics.databoard.Methods;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.method.TcpConnection.ConnectionListener;
import org.simantics.databoard.serialization.SerializationException;

/**
 * Proxy InterfaceBinding over a socket.
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class Client implements MethodInterface {
	
	Socket s;
	TcpConnection c;

	/**
	 * Create a new method interface client. On successful construction the 
	 * connection is established over TCP/IP channel.
	 * 
	 * @param addr
	 * @param port
	 * @throws IOException connection error
	 * @throws SerializationException handshake communication error
	 * @throws BindingException handshake communication error
	 */
	public Client(String addr, int port) 
	throws IOException, SerializationException, BindingException {
		this(InetAddress.getByName(addr), port);
	}

	/**
	 * Create a new method interface client. On successful construction the 
	 * connection is established over TCP/IP channel.
	 * 
	 * @param sa
	 * @throws IOException connection error
	 * @throws SerializationException handshake communication error
	 * @throws BindingException handshake communication error
	 */
	public Client(InetSocketAddress sa) 
	throws IOException, SerializationException, BindingException {
		s = new Socket(sa.getAddress(), sa.getPort());
		Handshake local = new Handshake();
		local.methods = Methods.noMethods().getInterface().getMethodDefinitions();
		Handshake remote = TcpConnection.handshake(s, local);
		c = new TcpConnection(s, Methods.noMethods(), local, remote);		
	}

	/**
	 * Create a new method interface client. On successful construction the 
	 * connection is established over TCP/IP channel.
	 * 
	 * @param addr
	 * @param port
	 * @throws IOException connection error
	 * @throws SerializationException handshake communication error
	 * @throws BindingException handshake communication error
	 */
	public Client(InetAddress addr, int port) 
	throws IOException, SerializationException, BindingException {
		s = new Socket(addr, port);
		Handshake local = new Handshake();
		local.methods = Methods.noMethods().getInterface().getMethodDefinitions();
		Handshake remote = TcpConnection.handshake(s, local);
		c = new TcpConnection(s, Methods.noMethods(), local, remote);		
	}
	
	/**
	 * Create a new method interface client. On successful construction the 
	 * connection is established over TCP/IP channel.
	 * 
	 * @param addr
	 * @param port
	 * @param localMethodHandler handles requests sent by the server
	 * @throws IOException
	 * @throws BindingException 
	 * @throws SerializationException 
	 */
	public Client(InetAddress addr, int port, MethodInterface localMethodHandler) 
	throws IOException, SerializationException, BindingException {
		s = new Socket(addr, port);
		Handshake local = new Handshake();
		local.methods = localMethodHandler.getInterface().getMethodDefinitions();
		Handshake remote = TcpConnection.handshake(s, local);
		c = new TcpConnection(s, localMethodHandler, local, remote);		
	}

	public void setConnectListener(ConnectionListener listener) 
	{
		c.addConnectionListener(listener);
	}
	
	/**
	 * Get remote method descriptions
	 */
	@Override
	public Interface getInterface() {
		return c.getInterface();
	}
	
	@Override
	public Method getMethod(MethodTypeBinding binding)
			throws MethodNotSupportedException {
		return c.getMethod(binding);
	}
	
	@Override
	public Method getMethod(MethodTypeDefinition description)
			throws MethodNotSupportedException {
		return c.getMethod(description);
	}
	
	public TcpConnection getConnection() 
	{
		return c;
	}
	
	public void close()
	{
		c.close();
		try {
			s.close();
		} catch (IOException e) {
		}
	}
	
}

