/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.file;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.RandomAccess;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.SerializerConstructionException;
import org.simantics.databoard.serialization.SerializerScheme;
import org.simantics.databoard.util.binary.BinaryFile;

/**
 * FileList is a file based implementation of a List collection.
 * add() serializes the object to a file and get() deserializes.<p>
 * 
 * Set, remove, insert and add operations flushes modifications to disk before
 * returning.<p>
 * 
 * Each operation may throw {@link RuntimeIOException}<p> 
 * 
 * Entry position index is on open if the file has variable width
 * data type (eg. String). The entire file is scanned through.<p>
 * 
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class FileList<T> extends RandomAccessBinaryList<T> implements IFileList<T>, RandomAccess {
	
	/** File Reference */
	File f;
	
	/** Random Access File */
	RandomAccessFile raf;

	/**
	 * Create new random access list of Variants (Object) backed by a file
	 * 
	 * @param file
	 * @throws IOException
	 * @throws BindingConstructionException 
	 * @throws SerializationException 
	 * @throws SerializerConstructionException 
	 */
	public FileList(File file) throws IOException, BindingConstructionException, SerializerConstructionException, SerializationException
	{
		this(file, Bindings.getBinding(Object.class), 0L, Bindings.serializationFactory);
	}
	
	/**
	 * Create new random access list backed by a file
	 * 
	 * @param file
	 * @param clazz
	 * @throws IOException
	 * @throws BindingConstructionException 
	 * @throws SerializationException 
	 * @throws SerializerConstructionException 
	 */
	public FileList(File file, Class<T> clazz) throws IOException, BindingConstructionException, SerializerConstructionException, SerializationException
	{
		this(file, Bindings.getBinding(clazz), 0L, Bindings.serializationFactory);
	}
	
	/**
	 * Create new random access list backed by a file
	 * 
	 * @param file
	 * @param clazz
	 * @throws IOException
	 * @throws BindingConstructionException 
	 * @throws SerializationException 
	 * @throws SerializerConstructionException 
	 */
	public FileList(String file, Class<T> clazz) throws IOException, BindingConstructionException, SerializerConstructionException, SerializationException
	{
		this(new File(file), Bindings.getBinding(clazz), 0L, Bindings.serializationFactory);
	}	

	/**
	 * Create new random access list backed by a file
	 * 
	 * @param file
	 * @param binding
	 * @throws IOException
	 * @throws SerializationException 
	 * @throws SerializerConstructionException 
	 */
	public FileList(String file, Binding binding) throws IOException, SerializerConstructionException, SerializationException
	{
		this(new File(file), binding, 0L, Bindings.serializationFactory);
	}
	
	/**
	 * Create new random access list backed by a file
	 * 
	 * @param file
	 * @param binding
	 * @throws IOException
	 * @throws SerializationException 
	 * @throws SerializerConstructionException 
	 */
	public FileList(File file, Binding binding) throws IOException, SerializerConstructionException, SerializationException
	{
		this(file, binding, 0L, Bindings.serializationFactory);
	}
	
	
	/**
	 * Create new random access list backed by a file
	 * 
	 * @param file
	 * @param clazz
	 * @param startPos
	 * @throws IOException
	 * @throws BindingConstructionException 
	 * @throws SerializationException 
	 * @throws SerializerConstructionException 
	 */
	public FileList(File file, Class<T> clazz, long startPos) 
	throws IOException, BindingConstructionException, SerializerConstructionException, SerializationException
	{
		this(file, Bindings.getBinding(clazz), startPos, Bindings.serializationFactory);
	}
	
	/**
	 * Create new random access list backed by a file using the default binary serialization.
	 * 
	 * @param file file
	 * @param binding
	 * @param startPos The position of the first sample in file
	 * @throws IOException 
	 * @throws SerializationException Error with the file, could not build entry index table
	 * @throws SerializerConstructionException 
	 */
	public FileList(File file, Binding binding, long startPos) 
	throws IOException, SerializationException, SerializerConstructionException
	{
		super(new BinaryFile(new RandomAccessFile(file, "rw")), binding, startPos, Bindings.serializationFactory);
		this.f = file;
		raf = ((BinaryFile)blob).getRandomAccessFile();
	}
	
	/**
	 * Create new random access list backed by a file
	 * 
	 * @param file file
	 * @param binding
	 * @param startPos The position of the first sample in file
	 * @param format serialization format
	 * @throws IOException 
	 * @throws SerializerConstructionException could not create serializer, never thrown with BinarySerializationFormat
	 * @throws SerializationException Error with the file, could not build entry index
	 */
	public FileList(File file, Binding binding, long startPos, SerializerScheme format) 
	throws IOException, SerializerConstructionException, SerializationException 
	{		
		super(new BinaryFile(new RandomAccessFile(file, "rw")), binding, startPos, format);
		f = file;
		raf = ((BinaryFile)blob).getRandomAccessFile();		
	}	
	
	/**
	 * Flushes the caches and closes the file handle. 
	 */
	public void close()
	{
		super.close();		
		if (raf!=null)
		try {
			raf.close();
		} catch (IOException ignored) {
		}
	}

	@Override
	public File getFile() {
		return f;
	}

}

