package org.simantics.databoard.binding.impl;

import org.simantics.databoard.binding.LongBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.primitives.UnsignedLong;
import org.simantics.databoard.type.LongType;

/**
 * Binding of {@link UnsignedLong} to long type.
 * This is bitwise binding, i.e. high values are negative.
 * 
 * @author Toni Kalajainen <toni.kalajainen@iki.fi>
 */
public abstract class UnsignedLongBinding extends LongBinding {

	UnsignedLongBinding(LongType type) {
		super(type);
	}
	
	public static class Immutable extends UnsignedLongBinding {

		public Immutable(LongType type) {
			super(type);
		}

		@Override
		public Object create(Number value) throws BindingException {
			return UnsignedLong.fromBits(value.longValue());
		}

		@Override
		public Object create(String value) throws BindingException {
			try {
				return new UnsignedLong.Immutable(value);
			} catch (java.lang.IllegalArgumentException e) {
				throw new BindingException( e );
			}
		}

		@Override
		public Object create(long value) throws BindingException {
			return UnsignedLong.fromBits(value);
		}

		@Override
		public Object create(Long value) throws BindingException {
			return UnsignedLong.fromBits(value);
		}

		@Override
		public void setValue(Object obj, long value) throws BindingException {
			throw new BindingException("UnsignedLong is immutable class");
		}

		@Override
		public void setValue(Object obj, Number value) throws BindingException {
			throw new BindingException("UnsignedLong is immutable class");
		}
		
		@Override
		public boolean isImmutable() {
			return true;
		}
		
		@Override
		public boolean isInstance(Object obj) {
			return obj instanceof UnsignedLong.Immutable;
		}
		
	}

	public static class Mutable extends UnsignedLongBinding {

		public Mutable(LongType type) {
			super(type);
		}

		@Override
		public Object create(Number value) throws BindingException {
			return UnsignedLong.Mutable.fromBits(value.longValue());
		}

		@Override
		public Object create(String value) throws BindingException {
			return new UnsignedLong.Mutable(value);
		}

		@Override
		public void setValue(Object obj, Number value) throws BindingException {
			UnsignedLong.Mutable v = (UnsignedLong.Mutable) obj;
			v.setValue(value.longValue());
		}
		
		@Override
		public boolean isImmutable() {
			return false;
		}

		@Override
		public Object create(long value) throws BindingException {
			return UnsignedLong.Mutable.fromBits(value);
		}

		@Override
		public Object create(Long value) throws BindingException {
			return UnsignedLong.Mutable.fromBits(value);
		}

		@Override
		public void setValue(Object obj, long value) throws BindingException {
			UnsignedLong.Mutable v = (UnsignedLong.Mutable) obj;
			v.setValue(value);
		}

		@Override
		public boolean isInstance(Object obj) {
			return obj instanceof UnsignedLong.Mutable;
		}
		
	}	
	
	@Override
	public String toString(Object value) throws BindingException {
		return value.toString();
	}

	@Override
	public Long getValue(Object o) throws BindingException {
		UnsignedLong v = (UnsignedLong) o;
		return v.toBits();
	}

	@Override
	public long getValue_(Object o) throws BindingException {
		UnsignedLong v = (UnsignedLong) o;
		return v.toBits();
	}

}
