/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.lang.reflect.Array;
import java.util.Iterator;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.adapter.Adapter;
import org.simantics.databoard.adapter.AdapterConstructionException;
import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.ArrayType;

/**
 * Binds to Object[]
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class ObjectArrayBinding extends ArrayBinding {

	public ObjectArrayBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
		if (type == null)
			throw new IllegalArgumentException("null arg");
		this.type = type;
	}

	public ObjectArrayBinding(Binding componentBinding) {
		this(new ArrayType(componentBinding.type()), componentBinding);
	}

	@Override
	public Object create(int length, Iterator<Object> values) {
		Object[] array = new Object[length];
		for (int i = 0; i < length; ++i)
			array[i] = values.next();
		return array;
	}

	@Override
	public Object create() {
		return new Object[0];
	}
	
	public Object create(Object[] values) {
		return values.clone();
	}

	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		Binding scb = sb.getComponentBinding();
		// Dst component binding
		Binding dcb = getComponentBinding();
		// Dst array
		Object[] d = (Object[]) dst;
		if (d.length != sb.size(src)) throw new BindingException("Object[] is length immutable");
		
		for (int i=0; i<d.length; i++) {
			Object sc = sb.get(src, i);
			Object dc = d[i];
			d[i] = dcb.readFromTry(scb, sc, dc);
		}
	}	
	
	@Override
	public Object readFromTry(Binding srcBinding, Object src, Object dst) throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		Binding scb = sb.getComponentBinding();
		// Dst component binding
		Binding dcb = getComponentBinding();
		// Dst array
		Object[] d = (Object[]) dst;
		int srcSize = sb.size(src);
		if (d.length != srcSize) {
			Object[] oldD = (Object[]) dst;
			d = new Object[ srcSize ];
			int X = Math.min(oldD.length, srcSize);
			for (int i=0; i<X; i++) {
				Object sc = sb.get(src, i);
				Object dc = d[i];
				d[i] = dcb.readFromTry(scb, sc, dc);
			}
			if (X<srcSize) {
				try {
					Adapter cloner = Bindings.adapterFactory.getAdapter(scb, dcb, false, true);
					for (int i=X; i<srcSize; i++) {
						Object sc = sb.get(src, i);
						Object dc = cloner.adapt( sc );
						d[i] = dc;
					}
				} catch (AdaptException e) {
					throw new BindingException( e );
				} catch (AdapterConstructionException e) {
					throw new BindingException( e );
				}
			}
		} else {
			for (int i=0; i<d.length; i++) {
				Object sc = sb.get(src, i);
				Object dc = d[i];
				d[i] = dcb.readFromTry(scb, sc, dc);			
			}
		}
		return d;
	}	
	
	@Override
	public void add(Object array, int index, Object element)
	throws BindingException, IndexOutOfBoundsException {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
		throw new UnsupportedOperationException();
	}

	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		Object[] list = (Object[]) array;
		System.arraycopy(list, 0, result, 0, list.length);
	}
	
	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array))
			throw new BindingException("Unexpected class "
					+ array.getClass().getSimpleName() + ", Object[] expected");
		return ((Object[]) array)[index];
	}
	
	@Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		Object[] list = (Object[]) array;
		list[index] = value;
	}

	@Override
	public int size(Object array) throws BindingException {
		if (!isInstance(array))
			throw new BindingException("Unexpected class "
					+ array.getClass().getSimpleName() + ", Object[] expected");
		return ((Object[]) array).length;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof Object[];
	}
	
	@Override
	public boolean isImmutable() {
		return false;
	}

	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int oldSize = Array.getLength(array);
		if (oldSize==newSize) return;
		throw new BindingException("Object[] is length immutable");
	}	
	
	@Override
	public boolean isResizable() {
		return false;
	}
	
}
