/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import org.simantics.databoard.binding.IntegerBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.IntegerType;

/**
 * Binds Integer Type to java.lang.Integer
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class IntegerBindingDefault extends IntegerBinding {

	public IntegerBindingDefault(IntegerType type) {
		super( type );
	}
	
	public Object create(int value) {
		return Integer.valueOf(value);
	}
	
	public Object create(Integer value) {
		return value;
	}
		
	@Override
	public Object create(Number value) 
	{
		if (value.getClass()==Integer.class) return value;
		return Integer.valueOf( value.intValue() );
	}

	@Override
	public Object create(String value) throws BindingException {
		try {
			return Integer.parseInt(value);
		} catch (java.lang.NumberFormatException e) {
			throw new BindingException( e );
		}
	}
	        	
	public Integer getValue(Object obj) 
	throws BindingException
	{
		if (obj.getClass()!=Integer.class)
			throw new BindingException("Unexpected class "+obj.getClass().getSimpleName()+", Integer expected");		
		return ((Integer)obj);
	}
	
	public int getValue_(Object obj)
	throws BindingException
	{
		if (obj.getClass()!=Integer.class)
			throw new BindingException("Unexpected class "+obj.getClass().getSimpleName()+", Integer expected");		
		return ((Integer)obj);
	}

	@Override
	public void setValue(Object obj, Number value) throws BindingException {
		throw new UnsupportedOperationException("Cannot change the value of immutable java.lang.Integer");
	}

	public void setValue(Object obj, int value) throws BindingException {
		throw new UnsupportedOperationException("Cannot change the value of immutable java.lang.Integer");
	}
	
	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof Integer;
	}	
	
	@Override
	public boolean isImmutable() {
		return true;
	}
	
    @Override
    public int compare(Object o1, Object o2)
    {
    	int i1 = (Integer) o1;
    	int i2 = (Integer) o2;
    	return (i1<i2 ? -1 : (i1==i2 ? 0 : 1));
    }
	
	
}

