/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.lang.reflect.Array;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.FloatBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.util.IdentityPair;

/**
 * Binds Float[] Type to float[] class.
 */
public final class FloatArrayBinding extends ArrayBinding {

	public static FloatArrayBinding createFrom(ArrayType type) {
		return new FloatArrayBinding(type, new FloatBindingDefault( (FloatType) type.componentType ));
	}

	public FloatArrayBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
	}

	@Override
	public Object create() {
		return new float[0];
	}
	
	@Override
	public Object create(int length, Iterator<Object> values) {
		float[] array = new float[length];
		for(int i=0;i<length;++i)
			array[i] = (Float)values.next();
		return array;
	}
	
	public Object create(Object[] values) {
		float[] result = new float[values.length];
		for (int i=0; i<values.length; i++)
			result[i] = (Float) values[i];
		return result;
	}		
	
	/**
	 * Create an array object.
	 * Note! The argument is consumed (is used in the result).
	 * 
	 * @param array
	 * @return an object that contains the array
	 */
	public Object create(float[] array) {
		return array;
	}

	@Override
    public Object createDefault()
    throws BindingException
    {
		NumberType nt = (NumberType) type().componentType;
		if (nt.getRange() == null) {
	    	return new float[ type().minLength() ];
		}
		return super.createDefault();
    }	
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		FloatBinding scb = (FloatBinding) sb.getComponentBinding();
		
		float[] d = (float[]) dst;
		if (d.length != sb.size(src)) throw new BindingException("float[] is length immutable");
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
	}
	
	@Override
	public Object readFromTry(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		FloatBinding scb = (FloatBinding) sb.getComponentBinding();
		
		float[] d = (float[]) dst;
		int srcSize = sb.size(src);
		if (d.length != srcSize) d = new float[ srcSize ];
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
		
		return d;
	}	
		
	
	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", byte[] expected");
		return ((float[])array)[index];
	}

	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		float[] a = (float[])array;
		for (int i=0; i<a.length; i++)
			result[i] = a[i];
	}
	
	@Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		float[] a = (float[])array;
		a[index] = (Float) value;
	}	
	
	@Override
	public int size(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", byte[] expected");
		return ((float[])array).length;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof float[];
	}
	
	@Override
	public boolean isImmutable() {
		return false;
	}
	
	public float[] getArray(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", byte[] expected");
		return (float[]) array;
	}	
	
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int result = 1;
    	float[] array = (float[]) value;
    	for (int i=0; i<array.length; i++) {
    		float element = array[i];    		
    		long bits = Float.floatToIntBits(element);
    		int elementHash = (int)(bits ^ (bits >>> 32));     		
    		result = 31*result + elementHash;
    	}
    	return result;
    }

    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	float[] a1 = (float[]) o1;
    	float[] a2 = (float[]) o2;
		// Compare Lengths
		int l1 = a1.length;
		int l2 = a2.length;
		int dif = l1 - l2;
		if (dif!=0) return dif;
		// Compare elements
		for (int i=0; i<l1; i++) {
			float v1 = a1[i];
			float v2 = a2[i];
			dif = Float.compare(v1, v2);
			if (dif!=0) return dif;
		}
		return 0;
    }    

    @Override
    public void add(Object array, int index, Object element)
    		throws BindingException, IndexOutOfBoundsException {
    	throw new UnsupportedOperationException();
    }
    
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
    	throw new UnsupportedOperationException();
	}
    
	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int oldSize = Array.getLength(array);
		if (oldSize==newSize) return;
		throw new BindingException("float[] is length immutable");
	}	

	@Override
	public boolean isResizable() {
		return false;
	}
	
}
