/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.ArrayType;

/**
 * CollectionListBinding binds ArrayType to java.util.List and 
 * instantiates java.util.ArrayList.
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class ArrayListBinding extends ArrayBinding {

	public ArrayListBinding(Binding componentBinding) {
		this(new ArrayType(componentBinding.type()), componentBinding);
	}
	
	public ArrayListBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
		if (type==null) throw new IllegalArgumentException("null arg");
		this.type = type;
	}
	
	@Override
	public Object create() {
		return new ArrayList<Object>();
	}
	
	@Override
	public Object create(Collection<Object> collection) throws BindingException {
		return new ArrayList<Object>( collection );
	}
	
	/**
	 * Create new ArrayList
	 */
	@Override
	public Object create(int length, Iterator<Object> values) {
		ArrayList<Object> array = new ArrayList<Object>(length);
		while (values.hasNext())
			array.add(values.next());
		return array;
	}

	public Object create(Object[] values) {
		ArrayList<Object> array = new ArrayList<Object>(values.length);
		for (int i=0; i<values.length; i++)
			array.add(values[i]);
		return array;
	}
	
	
	@SuppressWarnings("unchecked")
	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", java.util.List expected");
		List<Object> list = (List<Object>) array;
		return list.get(index);
	}
	
	@SuppressWarnings("unchecked")
    @Override
	public void getAll(Object array, Object[] result) throws BindingException {
		List<Object> list = (List<Object>) array;
		for (int i=0; i<list.size(); i++)
			result[i] = list.get(i);
	}
	
	@SuppressWarnings("unchecked")
    @Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		List<Object> list = (List<Object>) array;
		list.set(index, value);
	}

	@SuppressWarnings("unchecked")
    @Override
	public void add(Object array, int index, Object element)
			throws BindingException, IndexOutOfBoundsException {
		List<Object> list = (List<Object>) array;
		list.add(index, element);
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
		List<Object> list = (List<Object>) array;		
		for (int i=0; i<count; i++)
			list.remove(index);
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public int size(Object array) throws BindingException {		
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", java.util.List expected");		
		List<Object> list = (List<Object>) array;
		return list.size();
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof List<?>;
	}

	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		@SuppressWarnings("unchecked")
		List<Object> list = (List<Object>) array;
		int oldSize = list.size();
		if (oldSize==newSize) return;
		
		if (oldSize>newSize) {
			for (int i=oldSize-1; i>=newSize; i--)
				list.remove(i);
			return;
		} 
		
		int c = newSize - oldSize;
		for (int i=0; i<c; i++) {
			list.add( componentBinding.createDefault() );
		}
	}		
	
	@Override
	public boolean isImmutable() {
		return false;
	}
	
	@Override
	public boolean isResizable() {
		return true;
	}
	
}

