/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.factory;

import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.impl.ArrayListBinding;
import org.simantics.databoard.binding.impl.TreeMapBinding;
import org.simantics.databoard.binding.mutable.ContainerOptionalBinding;
import org.simantics.databoard.binding.mutable.MutableBooleanBinding;
import org.simantics.databoard.binding.mutable.MutableByteBinding;
import org.simantics.databoard.binding.mutable.MutableDoubleBinding;
import org.simantics.databoard.binding.mutable.MutableFloatBinding;
import org.simantics.databoard.binding.mutable.MutableIntegerBinding;
import org.simantics.databoard.binding.mutable.MutableLongBinding;
import org.simantics.databoard.binding.mutable.MutableStringBinding;
import org.simantics.databoard.binding.mutable.UnionTaggedObjectBinding;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.OptionalType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.type.UnionType;

/**
 * MutableBindingScheme is a type to binding mapping that binds any DataType to an Object.
 * All resulting bindings are completely mutable java classes.
 * 
 * DataType           | Class of the bound instance
 * ===================|==================
 * BooleanType        | MutableBoolean.class
 * ByteType           | MutableByte.class
 * FloatType          | MutableFloat.class
 * DoubleType         | MutableDouble.class
 * IntegerType        | MutableInt.class
 * LongType           | MutableLong.class
 * StringType         | MutableString.class
 * UnionType          | TaggedObject.class
 * OptionType         | ValueContainer.class
 * RecordType         | Object[].class
 * ArrayType          | ArrayList.class
 * MapType            | TreeMap.class
 * VariantType        | MutableVariant.class
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */

public class MutableBindingFactory extends TypeBindingFactory {

	/**
	 * Construct a binding factory.
	 */
	public MutableBindingFactory() {
		super();
	}
	
	/**
	 * Construct a scheme factory that appends constructed bindings to the user given
	 * repository  
	 * 
	 * @param repository repository where bindings are placed
	 */
	public MutableBindingFactory(Map<Datatype, Binding> repository) {
		super(repository);
	}

	@Override
	protected Binding doConstruct(Datatype type)
			throws BindingConstructionException {

		// Exact, non-annotated types
		if (type.equals( Bindings.BOOLEAN.type() )) return Bindings.MUTABLE_BOOLEAN;
		if (type.equals( Bindings.BYTE.type() )) return Bindings.MUTABLE_BYTE;
		if (type.equals( Bindings.INTEGER.type() )) return Bindings.MUTABLE_INTEGER;
		if (type.equals( Bindings.LONG.type() )) return Bindings.MUTABLE_LONG;
		if (type.equals( Bindings.FLOAT.type() )) return Bindings.MUTABLE_FLOAT;
		if (type.equals( Bindings.DOUBLE.type() )) return Bindings.MUTABLE_DOUBLE;
		if (type.equals( Bindings.STRING.type() )) return Bindings.MUTABLE_STRING;
		if (type.equals( Bindings.VARIANT.type() )) return Bindings.MUTABLE_VARIANT;
		
		// Annotated types 
		if (type instanceof BooleanType) return new MutableBooleanBinding((BooleanType)type);
		if (type instanceof DoubleType) return new MutableDoubleBinding((DoubleType)type);
		if (type instanceof FloatType) return new MutableFloatBinding((FloatType)type);
		if (type instanceof ByteType) return new MutableByteBinding((ByteType)type);
		if (type instanceof IntegerType) return new MutableIntegerBinding((IntegerType)type);
		if (type instanceof LongType) return new MutableLongBinding((LongType)type);
		if (type instanceof StringType) return new MutableStringBinding((StringType)type);
		
		// Constructed types
		if (type instanceof ArrayType) {
			ArrayType arrayType = (ArrayType) type;
			Datatype componentType = arrayType.componentType();

			ArrayListBinding binding = new ArrayListBinding(arrayType, null);
			inprogress.put(type, binding);
			binding.componentBinding = construct( componentType );						
			inprogress.remove(type);
			return binding;
		}
		
		if (type instanceof OptionalType) {
			OptionalType optionalType = (OptionalType) type;
			Datatype componentType = optionalType.componentType;
			ContainerOptionalBinding binding = new ContainerOptionalBinding( optionalType, null );
			inprogress.put(type, binding);
			binding.componentBinding = construct( componentType );
			inprogress.remove(type);
			return binding;
		}
		
		if (type instanceof RecordType) {			
			RecordType recordType = (RecordType) type;
			Binding componentBindings[] = new Binding[ recordType.getComponentCount() ];
			RecordObjectArrayBinding binding = new RecordObjectArrayBinding(recordType, componentBindings);
			inprogress.put(type, binding);
			for (int i=0; i<componentBindings.length; i++) {
				componentBindings[i] = construct( recordType.getComponentType(i) );
			}
			inprogress.remove(type);
			return binding;
		}
		
		if (type instanceof UnionType) {
			UnionType unionType = (UnionType) type;
			Binding componentBindings[] = new Binding[ unionType.components.length ];
			UnionTaggedObjectBinding binding = new UnionTaggedObjectBinding(unionType, componentBindings);
			inprogress.put(type, binding);
			for (int i=0; i<componentBindings.length; i++) {
				componentBindings[i] = construct( unionType.getComponent(i).type );
			}
			inprogress.remove(type);
			return binding;
		}		
		
		if (type instanceof MapType) {			
			MapType mapType = (MapType) type;
			TreeMapBinding binding = new TreeMapBinding(mapType, null, null);
			inprogress.put(type, binding);
			binding.setKeyBinding( construct(mapType.keyType) );
			binding.setValueBinding( construct(mapType.valueType) );
			inprogress.remove(type);
			return binding;
		}
		
		throw new BindingConstructionException("Unexpected, I don't know how to create binding for "+type);
	}

	@Override
	public boolean supportsType(Datatype type) {
		// unexpected
		if (failures.containsKey(type)) return false;
		return true;
	}
	
}






