/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.classfactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeMap;

import org.simantics.databoard.annotations.ArgumentImpl;
import org.simantics.databoard.annotations.LengthImpl;
import org.simantics.databoard.annotations.MIMETypeImpl;
import org.simantics.databoard.annotations.PatternImpl;
import org.simantics.databoard.annotations.RangeImpl;
import org.simantics.databoard.annotations.UnitImpl;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.binding.reflection.BindingRequest;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.type.VariantType;

public class ImmutableClassesFactory implements TypeClassSubFactory {
	
	public ImmutableClassesFactory() {
	}
	
	@Override
	public BindingRequest construct(TypeClassFactory mainFactory, Datatype type) 
	throws BindingConstructionException 
	{
		
		if ( type instanceof ArrayType ) {
			List<Annotation> annotations = new ArrayList<Annotation>();
			ArrayType at = (ArrayType) type;
			BindingRequest cbr = construct(mainFactory, at.componentType);
			if ( cbr == null ) {
				cbr = mainFactory.getClass(at.componentType);				
			}

			String length = at.metadata.get(ArrayType.KEY_LENGTH);
			if (length != null) annotations.add( new LengthImpl( length ) );
			
			if ( cbr.getClazz() != null ) {
				for (Annotation a : cbr.annotations)
				{
					annotations.add( a );
				}
				
				Class<?> arrayClass = Array.newInstance(cbr.getClazz(), 0).getClass();
				return new BindingRequest( arrayClass, annotations );
			}
		}
		
		if ( type instanceof MapType ) {
			MapType mt = (MapType) type;
			List<Annotation> annotations = new ArrayList<Annotation>();
			BindingRequest kbr = construct(mainFactory, mt.keyType);
			BindingRequest vbr = construct(mainFactory, mt.valueType);
			if ( kbr == null ) kbr = mainFactory.getClass(mt.keyType);							
			if ( vbr == null ) vbr = mainFactory.getClass(mt.valueType);
			if ( kbr.getClazz()!=null && vbr.getClazz()!=null ) {
				annotations.add( new ArgumentImpl(kbr.getClazz(), vbr.getClazz()) );			
				return new BindingRequest( TreeMap.class, annotations );
			}
		}
		
		if ( type instanceof StringType ) {
			List<Annotation> annotations = new ArrayList<Annotation>();
			StringType st = (StringType) type;
			
			String pattern = st.metadata.get(StringType.KEY_PATTERN);
			if (pattern != null) annotations.add( new PatternImpl( pattern ) );
			
			String mimetype = st.metadata.get(StringType.KEY_MIMETYPE);
			if (mimetype != null) annotations.add( new MIMETypeImpl( mimetype ) );
			
			String length = st.metadata.get(StringType.KEY_LENGTH);
			if (length != null) annotations.add( new LengthImpl( length ) );
			
			return new BindingRequest( String.class, annotations );			
		}
		
		if ( type instanceof NumberType ) {
			List<Annotation> annotations = new ArrayList<Annotation>();
			NumberType nt = (NumberType) type;
			
			String unit = nt.metadata.get(NumberType.KEY_UNIT);
			if (unit != null) annotations.add( new UnitImpl(unit) );
			
			String range = nt.metadata.get(NumberType.KEY_RANGE);
			if (range != null) annotations.add( new RangeImpl(range) );
			
			Class<?> clazz = null;
			if ( type instanceof IntegerType ) clazz = int.class;
			if ( type instanceof ByteType ) clazz = byte.class;
			if ( type instanceof LongType ) clazz = long.class;
			if ( type instanceof DoubleType ) clazz = double.class;
			if ( type instanceof FloatType ) clazz = float.class;
			
			return new BindingRequest( clazz, annotations );
		}
		
		if ( type instanceof BooleanType ) {
			return new BindingRequest( boolean.class );
		}
		
		if ( type instanceof VariantType ) {
			return new BindingRequest( Variant.class );
		}
		
		return null;
	}

}
