/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.accessor.reference.IndexReference;
import org.simantics.databoard.accessor.reference.LabelReference;
import org.simantics.databoard.accessor.reference.NameReference;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.impl.BindingPrintContext;
import org.simantics.databoard.type.Component;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.util.IdentityHashSet;
import org.simantics.databoard.util.IdentityPair;


/**
 * This is a binding of a Record Type and a Java Object.
 * 
 * @see RecordType
 * @author Hannu Niemisto
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public abstract class RecordBinding extends Binding {
    
    public Binding[] componentBindings;
    
    /**
     * Get binding by field name
     * @param fieldName
     * @return binding or <code>null</code>
     */
    public Binding getComponentBinding(String fieldName) {
    	int fieldIndex = type().getComponentIndex2(fieldName);
    	if (fieldIndex<0) return null;
    	return getComponentBinding(fieldIndex);
    }
    
    public Object getComponentObject(Object obj, String fieldName) throws BindingException {
    	int fieldIndex = type().getComponentIndex2(fieldName);
    	if (fieldIndex<0) return null;
    	return getComponent(obj, fieldIndex);
    }
        
    public int getComponentIndex(String fieldName) {
    	return type().getComponentIndex2(fieldName);
    }
    
    public Binding getComponentBinding(int fieldIndex) {
    	return componentBindings[fieldIndex];
    }
    
	public Binding[] getComponentBindings() {
	    return componentBindings;
	}
	
	protected void setComponentBindings(Binding[] componentBindings) {
		this.componentBindings = componentBindings;
	}
	
	@Override
	public RecordType type() {
		return (RecordType) type;
	}
	
	public int getComponentCount() {
		return type().getComponentCount();
	}
	
	public abstract Object getComponent(Object obj, int index)
	throws BindingException;
	
	/**
	 * Create a record using values.
	 * 
	 * Note! values may be consumed (used in the result)
	 * 
	 * @param values
	 * @return new record
	 * @throws BindingException
	 */
	public abstract Object create(Object ... values)
	throws BindingException;
	
	/**
	 * Creates partial and most likely invalid instance.
	 * This is used in two-phase construction of recursive instances.
	 *  
	 * @return instance.
	 */
	public abstract Object createPartial() throws BindingException;
	
	public abstract void setComponents(Object obj, Object ... value) throws BindingException;
	public abstract void setComponent(Object obj, int index, Object value) throws BindingException;

	public void setComponent(Object obj, String fieldName, Object value) throws BindingException
	{
		int fieldIndex = type().getComponentIndex2(fieldName);
		if ( fieldIndex<0 ) throw new BindingException("Field "+fieldName+" does not exist.");
		setComponent(obj, fieldIndex, value);
	}
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		RecordBinding sb = (RecordBinding) srcBinding;
		int len = getComponentCount();
		if (sb.getComponentCount()!=len) throw new BindingException("field count mismatch");
		try {
			for (int i=0; i<len; i++) {
				Binding dcb = getComponentBinding(i);
				Binding scb = sb.getComponentBinding(i);
				Object sc = sb.getComponent(src, i);
				if (dcb.isImmutable()) {
					Object cv = Bindings.clone(sc, scb, dcb);
					setComponent(dst, i, cv);
				} else {
					Object v = getComponent(dst, i);
					v = dcb.readFromTry(scb, sc, v);
					setComponent(dst, i, v);
				}
			}		
		} catch (AdaptException e) {
			throw new BindingException(e);
		}
	}
	
    @Override
    public void accept(Visitor1 v, Object obj) {
        v.visit(this, obj);        
    }
    
    @Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }
    
    /**
     * Assert obj is valid Record Type
     * 
     * This asserts all fields are valid. 
     * 
     * @throws BindingException
     */
    @Override
    public void assertInstaceIsValid(Object obj, Set<Object> validInstances) throws BindingException {
    	if (!isInstance(obj)) 
    		throw new BindingException("The object ("+obj+") is not correct instance.");
    	RecordType type = type();    	
    	int length = type.getComponentCount();
    	
    	if (type.isReferable()) {
    		if (validInstances==null) {
    			validInstances = new IdentityHashSet<Object>();
    		}
    		if (validInstances.contains(obj)) return;
    		validInstances.add(obj);
    	}
    	
    	for (int i=0; i<length; i++)
    	{
    		Binding componentBinding = getComponentBindings()[i]; 
    		Object componentValue = getComponent(obj, i);
    		componentBinding.assertInstaceIsValid(componentValue, validInstances);
    	}
    }
    
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	if (type().isReferable()) {
	    	if (hashedObjects==null) {
	    		hashedObjects = new IdentityHashMap<Object, Object>(1);
	    	}
	    	if (hashedObjects.containsKey(value)) {
	    		return 0;    	
	    	} else {
	    		hashedObjects.put(value, null);
	    	}
    	}
    	
    	int result = 3;
    	int len = componentBindings.length;
    	for (int i=0; i<len; i++) {
    		Object element = getComponent(value, i);
    		result += 31*result + componentBindings[i].deepHashValue(element, hashedObjects);
    	}
    	return result;
    }
    
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
		RecordType rt    = (RecordType) type(); 
		
		// Compare History is used to prevent infinite loops
		if (rt.isReferable()) {
        	if (compareHistory==null) {
        		compareHistory = new HashSet<IdentityPair<Object, Object>>(); 
        	} 	
        	IdentityPair<Object, Object> p = new IdentityPair<Object, Object>(o1, o2);
        	if (compareHistory.contains(p)) return 0;            	
    		compareHistory.add(p);
		}
		
 		int len = rt.getComponentCount();
		for (int i=0; i<len; i++) {
			Binding c = getComponentBindings()[i];
			Object v1 = getComponent(o1, i);
			Object v2 = getComponent(o2, i);
			int dif = c.deepCompare(v1, v2, compareHistory);
			if (dif!=0)
				return dif;
		}
		return 0;
    }
	
	public Object createUnchecked(Object ... values) throws RuntimeBindingException
	{
		try {
			return create(values);
		} catch (BindingException e) {
			throw new RuntimeBindingException(e);
		}
	}
	
	public void setComponentsUnchecked(Object obj, Object ... value) throws RuntimeBindingException {
		try {
			setComponents(obj, value);
		} catch (BindingException e) {
			throw new RuntimeBindingException(e);
		}
	}
	
	public void setComponentUnchecked(Object obj, int index, Object value) throws RuntimeBindingException {
		try {
			setComponent(obj, index, value);
		} catch (BindingException e) {
			throw new RuntimeBindingException(e);
		}
	}

	protected void toStringAux(Object value, BindingPrintContext ctx) throws BindingException {
		ctx.b.append('{');
		Component[] components = type().getComponents();
		boolean first = true;
		for(int i=0;i<components.length;++i) {
			if(first)
				first = false;
			else {
				ctx.b.append(", ");
				if (!ctx.singleLine) ctx.b.append('\n');
			}
			Binding binding = getComponentBinding(i);
			Object cval = getComponent(value, i);
			if(binding instanceof OptionalBinding && 
					!((OptionalBinding)binding).hasValue(cval))
				continue;
			Component component = components[i];			
			ctx.b.append(component.name);
			ctx.b.append(" = ");
			binding.toString(cval, ctx);
		}
		ctx.b.append('}');
	}
	
    @Override
	protected void toString(Object value, BindingPrintContext ctx) throws BindingException {
    	if(type().isReferable()) {
    		if(ctx.refs.contains(value)) {
    			int val = ctx.refs.get(value);
    			if(val < 0) {
    				val = ctx.nameCount.value++;
    				ctx.refs.put(value, val);
    			}
    			ctx.b.append((char)('a' + val));
    		}
    		else {
    			ctx.refs.put(value, -1);
    			toStringAux(value, ctx);
    			int val = ctx.refs.remove(value);
    			if(val >= 0) {
    				ctx.b.append("/");
    				ctx.b.append((char)('a' + val));
    			}
    		}
    	}
    	else {
    		toStringAux(value, ctx);
    	}
	}

	@Override
	public Binding getComponentBinding(ChildReference path) throws IllegalArgumentException {
		if (path==null) return this;
		if (path instanceof IndexReference) {
			IndexReference ir = (IndexReference) path;
			return componentBindings[ir.index].getComponentBinding(path.childReference);
		}
		if (path instanceof NameReference) {
			NameReference nr = (NameReference) path;
			return getComponentBinding( nr.name ).getComponentBinding(path.childReference);
		}
		if (path instanceof LabelReference) {
			LabelReference lr = (LabelReference) path;			
			try {
				Integer i = new Integer(lr.label);
				return getComponentBinding( i ).getComponentBinding(path.childReference);
			} catch (NumberFormatException nfe) {
				return getComponentBinding( lr.label ).getComponentBinding(path.childReference);
			}
		}
		throw new IllegalArgumentException();
	}
	
	@Override
	public boolean isImmutable() {
		return getComponentCount() == 0;
	}
	
	public void setBoolean(Object r, int index, boolean z) throws BindingException
	{
		setComponent(r, index, ((BooleanBinding)componentBindings[index]).create(z));
	}
	
	public boolean getBoolean(Object r, int index) throws BindingException
	{
		return ((BooleanBinding)componentBindings[index]).getValue_( getComponent(r, index) );
	}
	
	public void setByte(Object r, int index, byte x) throws BindingException
	{
		setComponent(r, index, ((ByteBinding)componentBindings[index]).create(x));
	}
	
	public byte getByte(Object r, int index) throws BindingException
	{
		return ((ByteBinding)componentBindings[index]).getValue_( getComponent(r, index) );
	}

	public void setInt(Object r, int index, int x) throws BindingException
	{
		setComponent(r, index, ((IntegerBinding)componentBindings[index]).create(x));
	}
	
	public int getInt(Object r, int index) throws BindingException
	{
		return ((IntegerBinding)componentBindings[index]).getValue_( getComponent(r, index) );
	}
	
	public void setLong(Object r, int index, long x) throws BindingException
	{
		setComponent(r, index, ((LongBinding)componentBindings[index]).create(x));
	}
	
	public long getLong(Object r, int index) throws BindingException
	{
		return ((LongBinding)componentBindings[index]).getValue_( getComponent(r, index) );
	}
	
	public void setFloat(Object r, int index, float x) throws BindingException
	{
		setComponent(r, index, ((FloatBinding)componentBindings[index]).create(x));
	}
	
	public float getFloat(Object r, int index) throws BindingException
	{
		return ((FloatBinding)componentBindings[index]).getValue_( getComponent(r, index) );
	}
	
	public void setDouble(Object r, int index, double x) throws BindingException
	{
		setComponent(r, index, ((DoubleBinding)componentBindings[index]).create(x));
	}
	
	public double getDouble(Object r, int index) throws BindingException
	{
		return ((DoubleBinding)componentBindings[index]).getValue_( getComponent(r, index) );
	}

	@Override
	protected boolean deepEquals(Object obj,
			Set<IdentityPair<Binding, Binding>> compareHistory) {
		RecordBinding o = (RecordBinding)obj;
		if (!super.deepEquals( obj, compareHistory ))
		    return false;
		
		if (componentBindings.length != o.componentBindings.length) return false;
		
		for (int i = 0; i < componentBindings.length; i++)
			if (!componentBindings[i].equals(o.componentBindings[i], compareHistory))
				return false;
		
		return true;
	}

	@Override
	public int deepHashCode(IdentityHashMap<Object, Object> hashedObjects) {
		int code = super.deepHashCode(hashedObjects);
		for (int i = 0; i < componentBindings.length; i++)
			code = 17 * code + componentBindings[i].hashCode(hashedObjects);
		return code;
	}
}
