/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.Set;

import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.impl.BindingPrintContext;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.util.Range;

/**
 * Super class for number types. 
 *
 * @author Toni Kalajainen
 */
public abstract class NumberBinding extends Binding {

	/**
	 * Create value by converting it from any Number instance to a Number 
	 * instance of this Binding type.
	 * 
	 * NOTE WARNING! Using this method may lose precision or value in the conversion. 
	 *   E.g. Double to Integer, or Long to Byte
	 * 
	 * @param value
	 * @return the value in the format of the binding type
	 */
	public abstract Object create(Number value)
	throws BindingException;
	
	/**
	 * Creates a value from its string representation
	 * @param value
	 * @return number
	 */
	public abstract Object create(String value)
	throws BindingException;
	
	/**
	 * Get Data type
	 * 
	 * @return data type
	 */
	@Override
	public NumberType type() {
		return (NumberType) type;
	}
	
	/**
	 * Get numeric value of an object
	 * 
	 * @param obj object
	 * @return Number
	 * @throws BindingException thrown if obj is incorrect class
	 */
	public Number getValue(Object obj) throws BindingException {
		return ((Number)obj);
	}
	
	public abstract void setValue(Object obj, Number value) throws BindingException;
	
	/**
	 * Assert the obj is a valid number. 
	 * 
	 * This asserts 
	 *  1. The value is within the range defined in the NumberType
	 * 
	 * @throws BindingException
	 */
	@Override
	public void assertInstaceIsValid(Object obj, Set<Object> validInstances) throws BindingException {
		if (!isInstance(obj)) throw new BindingException("Not a correct instance");		
		NumberType type = type();
		Range range = type.getRange();
		if (range!=null) {
			Number value = getValue(obj);
			if (!range.contains(value)) {
				throw new BindingException("Value ("+value+") out of range ("+range+")");
			}
		}
	}
	
	public Object createUnchecked(String value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	public Object createUnchecked(Number value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			return new RuntimeBindingException(e);
		}
	}
	
	@Override
	protected void toString(Object value, BindingPrintContext ctx) throws BindingException {
		ctx.b.append(getValue(value).toString());
	}
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		Number v = ((NumberBinding)srcBinding).getValue(src);
		setValue(dst, v);		
	}
	
	@Override
	public Binding getComponentBinding(ChildReference path) {
		if (path==null) return this;		
		throw new IllegalArgumentException();
	}	
	
	@Override
	public int getComponentCount() {
		return 0;
	}
	
	@Override
	public Binding getComponentBinding(int index) {
		throw new IllegalArgumentException();
	}
	
}

