/*******************************************************************************
 * Copyright (c) 2010- Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *    VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.wire;

import java.util.LinkedList;

import org.simantics.databoard.accessor.event.Event;
import org.simantics.databoard.accessor.interestset.InterestSet;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.binding.mutable.MutableVariant;
import org.simantics.databoard.type.Datatype;

public interface IWireServer {
	
	/**
	 * Open a handle to an accessor. 
	 * 
	 * @param ref accessor refence or <code>null<code> for root
	 * @return accessor info
	 * @throws WireException
	 */
	AccessorInfo openAccessor(ChildReference ref) throws WireException;
	
	static class AccessorInfo {
		public int accessorId;
		public Datatype type;
	}
	
	/**
	 * Close a group of accessor handles.
	 * Open listener and lock handles are released.
	 * 
	 * @param accIds accessor ids
	 */
	int closeAccessors(Integer[] accIds) throws WireException;
	
	/**
	 * Acquire mutual-exclusive write lock, and optionally read lock.
	 * The lock covers the node and its sub-tree.
	 * 
	 * Lock prevents access from other wire clients.
	 * 
	 * @param accId
	 * @param readLock
	 * @param timeout in milliseconds, -1 = infinite
	 * @return lockId, or -1 if failed 
	 */
//	int requestLock(int accId, boolean readLock, long timeout)  throws WireException;
	
	/**
	 * Release lock
	 * 
	 * @param lockId
	 */
//	void releaseLock(int lockId);

	/**
	 * Read the value as whole.
	 * 
	 * If accessor is locked, read operation delays until lock
	 * is released.
	 * 
	 * @param accId
	 * @return value
	 * @throws WireException 
	 */
	MutableVariant getValue(int accId) throws WireException;
	
	/**
	 * Apply a list of changes to the node.
	 * 
	 * If rollback is true, a rollback log is returned.
	 * Rollback log is a list of events that revert modifications
	 * if applied.<p> 
	 * 
	 * In case of error rollback log is filled with
	 * events that counter modification that were already applied
	 * before the error occured.<p> 
	 * 
	 * @param accId
	 * @param changeSet
	 * @param rollback
	 * @return apply result
	 */
	ApplyResult apply(int accId, Event[] changeSet, boolean rollback);
	
	public static class ApplyResult {
		public @Optional WireException error;
		public @Optional LinkedList<Event> rollbackLog;
	}

	/**
	 * Create listener 
	 * 
	 * @param accId
	 * @param interestSet
	 * @param pathPrefix
	 * @return lisId
	 * @throws WireException
	 */
	int addListener(int accId, InterestSet interestSet, ChildReference pathPrefix) throws WireException;
	
	/**
	 * Remove listener 
	 * 
	 * @param lisId
	 * @throws WireException
	 */
	int removeListener(int lisId) throws WireException;
	
	// Array, Map, Record, Union
	/**
	 * Get size of an array, map, record or union
	 * 
	 * @param accId
	 * @throws WireException
	 */
	int size(int accId) throws WireException;
	
	/**
	 * Remove all array or map elements
	 * 
	 * @param accId
	 * @throws WireException
	 */
	int clear(int accId) throws WireException;

	boolean containsKey(int accId, MutableVariant key) throws WireException;
	boolean containsValue(int accId, MutableVariant value) throws WireException;
	MutableVariant getFirstKey(int accId) throws WireException;
	MutableVariant getLastKey(int accId) throws WireException;
	MutableVariant getLowerKey(int accId, MutableVariant key) throws WireException;
	MutableVariant getFloorKey(int accId, MutableVariant key) throws WireException;
	MutableVariant getCeilingKey(int accId, MutableVariant key) throws WireException;
	MutableVariant getHigherKey(int accId, MutableVariant key) throws WireException;

	/**
	 * Get map entry value
	 *  
	 * @param accId
	 * @param key
	 * @return value
	 * @throws WireException
	 */
	MutableVariant getMapValue(int accId, MutableVariant key) throws WireException;
	
	/**
	 * Get map values as array
	 * 
	 * @param accId
	 * @return values in array inside a variant
	 * @throws WireException
	 */
	MutableVariant getMapValues(int accId) throws WireException;
	
	/**
	 * Get map keys as array
	 * 
	 * @param accId
	 * @return keys in array inside a variant
	 * @throws WireException
	 */
	MutableVariant getMapKeys(int accId) throws WireException;
	
	// Optional
	/**
	 * Get value status of an Optional node
	 * 
	 * @param accId
	 * @return <code>true</code> if there is value in the node 
	 * @throws WireException 
	 */
	boolean hasValue(int accId) throws WireException;
	
	// Union
	/**
	 * Get tag index of a Union node
	 * 
	 * @param accId
	 * @return tag index
	 * @throws WireException 
	 */
	int getTag(int accId) throws WireException;
	
	// Array
	
	/**
	 * add objects to array
	 * 
	 * @param accId
	 * @param index insert pos, -1 = at the end
	 * @param array
	 * @return actual insert pos
	 */
	int addAll(int accId, int index, MutableVariant array) throws WireException;
	
	/**
	 * 
	 * @param accId
	 * @param index insert pos, -1 = at the end
	 * @param object
	 * @return actual insert pos
	 * @throws WireException
	 */
	int add(int accId, int index, MutableVariant object) throws WireException;
	
	/**
	 * Get array element value
	 *  
	 * @param accId
	 * @param index
	 * @return value
	 * @throws WireException
	 */
	MutableVariant getArrayElement(int accId, int index) throws WireException;	
	
}

