/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.interestset;

import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.type.RecordType;

/**
 * Interest set of a Record Type.
 *
 * @see RecordType
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class RecordInterestSet extends InterestSet {

	// Notifications
	/** Interested in the _change_ to field assignments (excludes the value) */
	public boolean notification;
	/** Interested in the _change_ by field, null array/element if no interest */  
	public @Optional boolean[] notifications;
	
	// Value Assignments
	/** Interested in all values */ 
	public boolean value;
	/** Interested in values by field */
	public @Optional boolean[] values;
	
	// Content
	/** Component Interests by field, null array/element if no interest. */ 
	public @Optional InterestSet[] componentInterests;
	
	public RecordInterestSet(boolean notification, boolean[] specificNotifications, boolean value, boolean[] specificValues, InterestSet[] componentInterests) {
		this.notification = notification;
		this.notifications = specificNotifications;
		this.value = value;
		this.values = specificValues;
		this.componentInterests = componentInterests;
	}
	
	/**
	 * Returns true, if interested in notifications of changes to any field assignment
	 * 
	 * @return true, if interested in notifications
	 */
	public boolean inNotifications() {
		return notification | value;
	}

	/**
	 * Returns true, if interested in assignment of the field at <code>fieldIndex</code>
	 * 
	 * @param fieldIndex
	 * @return true, if interested in notifications of the field at <code>fieldIndex</code>
	 */
	public boolean inNotificationsOf(int fieldIndex) {
		if (notification || value) return true;		
		if (notifications != null && notifications[fieldIndex]) return true;
		if (values != null && values[fieldIndex]) return true;
		if (componentInterests!=null) {
			InterestSet compositeInterest = componentInterests[fieldIndex];
			return compositeInterest!=null;
		}
		return false;
	}
	
	/**
	 * Returns true, if interested in value of all elements 
	 * 
	 * @return true, if interested in values of all elements
	 */
	public boolean inValues() {
		return value;
	}

	/**
	 * Returns true, if interested in the complete value of the field at
	 * <code>fieldIndex</code>.
	 * 
	 * @param fieldIndex
	 * @return true, if interested in the values of the field at <code>fieldIndex</code>
	 */
	public boolean inValuesOf(int fieldIndex) {
		if (value) return true;
		if (values!=null && values[fieldIndex]) return true;
		if (componentInterests!=null) {
			InterestSet compositeInterest = componentInterests[fieldIndex];
			return compositeInterest!=null;
		}
		return false;
	}

	/**
	 * Get composite intrest for the element at <code>fieldIndex</code>
	 * 
	 * @param fieldIndex
	 * @return interest of component or <code>null</code>
	 */
	public InterestSet getComponentInterest(int fieldIndex) {
		if (componentInterests==null) return null;
		return componentInterests[fieldIndex];		
	}
	
	/**
	 * Return true there is notification interest to any component
	 * 
	 * @return true if there is interest in changes of components 
	 */
	public boolean inComponentNotifications() {
		return (notifications!=null && notifications.length>0) || (values!=null && values.length>0) || (componentInterests!=null && componentInterests.length>0);
	}
	
}

