/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.interestset;

import java.util.Map;
import java.util.TreeMap;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.MapType;

/**
 * Interest set of a Map Type. 
 *
 * @see MapType
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class MapInterestSet extends InterestSet {
	
	// Notifications
	/** Interested in the notifications of entry assignments, add and removal (excludes the value) */
	public boolean notification; // Assignment Notification
	/** Interested in the notifications of entry assignments by key (excludes the value). Sorted array of keys */  
	public @Optional Variant[] notifications;
	
	// Value Assignments
	/** Interested in new values of entry assignments (includes notification, excludes composite changes)*/ 
	public boolean value; // Assignments
	/** Interested in new values of entry assignments by key (includes notification, excludes composite changes). Sorted array of keys */
	public @Optional Variant[] values;
	
	// Content
	/** Interest in the contents of values (includes assignment notification and new values) */ 
	public @Optional InterestSet componentInterest; // Values
	/** Interest in the contents of values by key (includes assignment notification and new values) */ 
	public @Optional Map<Variant, InterestSet> componentInterests;

	public MapInterestSet() {}
	
	/**
	 * Create a map interest set
	 * 
	 * @param notification interested in add, remove and change notifications of entry assignments 
	 * @param notifications interested in add, remove and change to assignment of all specific entries
	 * @param value interested in add, remove and change to assignment of all entries, incl. new values
	 * @param values interested in add, remove and change to assignment of all specific entries, incl. new values
	 * @param componentInterest Interest of the changes of values and their sub-value changes of all entries
	 * @param componentInterests Interest of the changes of values and their sub-value changes of specific entries
	 */
	public MapInterestSet(boolean notification, Variant[] notifications, boolean value, Variant[] values, InterestSet componentInterest, Map<Variant, InterestSet> componentInterests) {
		this.notification = notification;
		this.notifications = notifications;
		this.value = value;
		this.values = values;
		this.componentInterest = componentInterest;
		this.componentInterests = componentInterests;
	}

	/**
	 * Returns true, if interested in notification of entry add, removal or replacement of container. 
	 * This doesn't include interest in the new value 
	 * 
	 * @return true if interested in notifications
	 */
	public boolean inNotifications() {
		return notification | value | (componentInterest!=null);
	}

	/**
	 * Returns true, if interested in notification of entry add, removal or replacement of container of
	 * specific keys. This doesn't include interest in the new value.
	 * 
	 * @return true if interested in notifications of a specific entry
	 */
	public boolean inNotificationsOf(Variant key) {
		if (notification || value) return true;

		if (notifications!=null) {
			for (int i=0; i<notifications.length; i++) {
				if (notifications[i] == key) return true;
			}
		}		
		
		if (values!=null) {
			for (int i=0; i<values.length; i++) {
				if (values[i] == key) return true;
			}
		}		
		
		if (componentInterests!=null) {
			InterestSet compositeInterest = componentInterests.get(key);
			return compositeInterest!=null;
		}
		return false;
	}
	
	/**
	 * Returns true, if interested in notifications and values of entry assignments. 
	 * 
	 * @return true, if interested in values of entry assignments.
	 */
	public boolean inValues() {
		return value | (componentInterest!=null);
	}

	/**
	 * Returns true, if interested in notifications and values of entry assignments of 
	 * <code>key</code>.
	 * 
	 * @return true, if interested in values of specific entry
	 */
	public boolean inValuesOf(Variant key) {
		if (value | componentInterest!=null) return true;

		if (values!=null) {
			for (int i=0; i<values.length; i++) {
				if (values[i] == key) return true;
			}
		}
		
		if (componentInterests!=null?componentInterests.containsKey(key):false) return true;
		return false;
	}

	/**
	 * Get the interest of component type of every value.  
	 * 
	 * @return interest of component or <code>null</code>
	 */
	public InterestSet getComponentInterest() {
		return componentInterest;
	}
	
	/**
	 * Get the interest of component type of entries of <code>key</code>.
	 * 
	 * @return interest of component or <code>null</code>
	 */
	public InterestSet getComponentInterest(Variant key) {
		if (componentInterests==null) return null;
		return componentInterests.get(key);		
	}
	
	/**
	 * Add key specific interest. Map is created if null.
	 * 
	 * @param key
	 * @param cis
	 */
	public void addCompositeInterest(Variant key, InterestSet cis) {
		if (componentInterests == null) componentInterests = new TreeMap<Variant, InterestSet>( Bindings.VARIANT );
		componentInterests.put(key, cis);
	}
	
}

