/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import org.simantics.databoard.accessor.Accessor;
import org.simantics.databoard.accessor.event.Event;
import org.simantics.databoard.accessor.event.ModificationEvent;

/**
 * ChangeSet is a collection of a modification events. ChangeSet can be applied
 * to an accessor, or captured from an accessor. To capture a change set
 * add the ChangeSet instance as a listener to the accessor. ChangeSet does not 
 * capture non-modification events.
 *
 * @see ModificationEvent
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class ChangeSet implements Accessor.Listener {

	public List<Event> events = new LinkedList<Event>();
	
	public synchronized void addEvent(Event event) {
		if (event instanceof ModificationEvent == false) return; 
		this.events.add(event);
	}

	public synchronized void addEvents(Collection<Event> events) {
		for (Event e : events) {
			if (e instanceof ModificationEvent == false) return; 
			this.events.addAll(events);
		}
	}
		
	/**
	 * Returns events and clears change set.
	 * 
	 * @return the list of events
	 */
	public synchronized List<Event> getAndClearEvents() {
		List<Event> result = events;
		events = new ArrayList<Event>();
		return result;
	}
	
	public synchronized boolean isEmpty() {
		return events.isEmpty();
	}
	
	/**
	 * Get a snapshot of events.
	 * 
	 * @return a copy of events
	 */
	public synchronized List<Event> getEvents() {
		return new ArrayList<Event>( events );
	}

	@Override
	public void onEvents(Collection<Event> events) {
		addEvents(events);
	}
	
	@Override
	public synchronized String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append('[');
		int len = events.size();
		if (len>1) sb.append("\n ");
		for (int i=0; i<len; i++) {
			Event e = events.get(i);
			if (i>0) sb.append("\n ");
			sb.append(e.toString());
		}
		if (len>1) sb.append('\n');
		sb.append(']');
		return sb.toString();
	}

	public synchronized void clear() {
		events.clear();
	}
	
}

