/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor;

import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.RecordType;

/**
 * Accessor to a Record.
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public interface RecordAccessor extends Accessor {

	/**
	 * Get the number of fields in the record
	 * 
	 * @return field count
	 * @throws AccessorException
	 */
	int count() throws AccessorException;

	/**
	 * Get an accessor to a field. The return value becomes invalid if a new 
	 * value is assigned with {@link #setFieldValue(int, Binding, Object)}.
	 * 
	 * @param index
	 * @return accessor to the field
	 * @throws AccessorConstructionException 
	 */
	<T extends Accessor> T getFieldAccessor(int index) throws AccessorConstructionException;

	/**
	 * Get an accessor to a field. The return value becomes invalid if a new 
	 * value is assigned with {@link #setFieldValue(int, Binding, Object)}.
	 * 
	 * @param fieldName
	 * @return accessor to the field
	 * @throws AccessorConstructionException 
	 */
	<T extends Accessor> T getFieldAccessor(String fieldName) throws AccessorConstructionException;
	
	/**
	 * Get field value.
	 * 
	 * Note, use RecordAccessor#type().getComponentIndex( fieldName ) to get index. <p>
	 * 
	 * @param index field index
	 * @return the value of the field
	 * @throws AccessorException
	 */
	Object getFieldValue(int index, Binding fieldBinding) throws AccessorException;
	
	/**
	 * Get field value.
	 * 
	 * Note, use RecordAccessor#type().getComponentIndex( fieldName ) to get index. <p>
	 * 
	 * @param fieldName 
	 * @return the value of the field
	 * @throws AccessorException
	 */
	Object getFieldValue(String fieldName, Binding fieldBinding) throws AccessorException;
		
	/**
	 * Set field value. 
	 *  
	 * Writing the current value again may not emit an event. This is implementation
	 * specific. <p>
	 * 
	 * @param index field index
	 * @param fieldBinding 
	 * @param value
	 * @throws AccessorException
	 */
	void setFieldValue(int index, Binding fieldBinding, Object value) throws AccessorException;

	/**
	 * Set field value. 
	 *  
	 * Writing the current value again may not emit an event. This is implementation
	 * specific. <p>
	 * 
	 * @param fieldName
	 * @param fieldBinding 
	 * @param value
	 * @throws AccessorException
	 */
	void setFieldValue(String fieldName, Binding fieldBinding, Object value) throws AccessorException;
	
	RecordType type();
	
}

