package org.simantics.acorn.lru;

import org.simantics.acorn.AcornKey;
import org.simantics.acorn.FileCache;
import org.simantics.acorn.exception.AcornAccessVerificationException;
import org.simantics.acorn.exception.IllegalAcornStateException;
import org.simantics.db.Database.Session.ResourceSegment;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.list.array.TByteArrayList;

public class FileInfo extends LRUObject<String, FileInfo> {

    private static final Logger LOGGER = LoggerFactory.getLogger(FileInfo.class);
	private TByteArrayList bytes;
	
	// Stub
	public FileInfo(LRU<String, FileInfo> LRU, FileCache fileCache, AcornKey readDir, String id, int offset, int length) throws AcornAccessVerificationException {
		super(LRU, fileCache, id, readDir, id.toString() + ".extFile", offset, length, false, false);
		LRU.map(this);
	}
	
	// New
	public FileInfo(LRU<String, FileInfo> LRU, FileCache fileCache, String id, int size) throws AcornAccessVerificationException {
		super(LRU, fileCache, id, LRU.getDirectory(), id.toString() + ".extFile", true, true);
		this.bytes = new TByteArrayList(size);
		LRU.insert(this, accessTime);
	}

	public byte[] getResourceFile() throws AcornAccessVerificationException, IllegalAcornStateException {
		
		if(VERIFY) verifyAccess();
		
		makeResident();
		return bytes.toArray();
	}
	
	
	public ResourceSegment getResourceSegment(final byte[] clusterUID, final int resourceIndex, final long segmentOffset, short segmentSize) throws AcornAccessVerificationException, IllegalAcornStateException {

		if(VERIFY) verifyAccess();

		makeResident();
		try {
			int segSize = segmentSize;
			if (segSize < 0)
				segSize += 65536;
			if (segmentSize == -1)
				segSize = Math.min(65535, bytes.size());

			final long valueSize = bytes.size();
			final byte[] segment = bytes.toArray((int) segmentOffset, segSize);

			return new ResourceSegment() {

				@Override
				public long getValueSize() {
					return valueSize;
				}

				@Override
				public byte[] getSegment() {
					return segment;
				}

				@Override
				public int getResourceIndex() {
					return resourceIndex;
				}

				@Override
				public long getOffset() {
					return segmentOffset;
				}

				@Override
				public byte[] getClusterId() {
					return clusterUID;
				}
			};
		} catch (Throwable t) {
			throw new IllegalAcornStateException(t);
		}
	}
	
	public void updateData(byte[] newBytes, long offset, long pos, long size) throws AcornAccessVerificationException, IllegalAcornStateException {

		if(VERIFY) verifyAccess();
		makeResident();

		if(size == 0) {
			bytes.remove((int)offset, (int)(bytes.size()-offset));
		} else {
			bytes.fill((int) (offset + size), (int) (offset + size), (byte) 0);
			bytes.set((int) offset, newBytes, (int) pos, (int) size);
		}
		
		setDirty();
		
	}
	
	@Override
	public Pair<byte[], Integer> toBytes() {
		byte[] result = bytes.toArray();
		release();
		return Pair.make(result, result.length);
	}
	
	@Override
	protected void release() {
		bytes = null;
	}

	@Override
	public void fromFile(byte[] data) {
		bytes = new TByteArrayList(data);
	}

	@Override
	protected String getExtension() {
		return "extFile";
	}
	
	@Override
	protected boolean overwrite() {
		return true;
	}

	@Override
	public Logger getLogger() {
	    return LOGGER;
	}
}