package org.simantics.acorn.lru;

import org.simantics.acorn.ClusterManager;
import org.simantics.acorn.cluster.ClusterImpl;
import org.simantics.acorn.exception.AcornAccessVerificationException;
import org.simantics.acorn.exception.IllegalAcornStateException;
import org.simantics.acorn.internal.ClusterUpdateProcessor;
import org.simantics.db.service.Bytes;
import org.simantics.db.service.ClusterUID;

final public class ClusterUpdateOperation {
	
	final public ClusterUID uid;
	final protected ClusterManager manager;
	final protected ClusterInfo info;

	public byte[] data;
	
	public ClusterStreamChunk chunk;
	public ClusterChangeSet ccs;
	boolean finished = false;

	public ClusterUpdateOperation(ClusterManager manager, byte[] data) throws IllegalAcornStateException, AcornAccessVerificationException {
		
		long cuid1 = Bytes.readLE8(data, 8);
		long cuid2 = Bytes.readLE8(data, 16);

		this.manager = manager;
		this.uid = ClusterUID.make(cuid1, cuid2);
		this.data = data;
		this.info = manager.clusterLRU.getOrCreate(uid, true);
	}
	
	public void finish() {
		finished = true;
	}
	
	public void scheduled(String ccsInfoId) throws AcornAccessVerificationException, IllegalAcornStateException {
		ccs = new ClusterChangeSet(ccsInfoId, uid);
		chunk = ccs.getChunk(manager);
		manager.addIntoCurrentChangeSet(ccsInfoId);
	}
	
	public void run() throws AcornAccessVerificationException, IllegalAcornStateException {
		ClusterUpdateOperation op = null;
		byte[] data = null;
		chunk.acquireMutex();
		try {
			chunk.makeResident();
			op = chunk.operations.get(ccs.chunkOffset);
			data = op.data;
		} finally {
			chunk.releaseMutex();
		}
		op.runWithData(data);
	}
	
	public void runWithData(byte[] data) throws IllegalAcornStateException, AcornAccessVerificationException {
		try {
			ClusterUpdateProcessor processor = new ClusterUpdateProcessor(manager, manager.support, data, this);
			ClusterImpl cluster = info.getForUpdate();
			cluster = processor.process(cluster);
			manager.update(uid, cluster);
		} catch (IllegalAcornStateException | AcornAccessVerificationException e) {
		    throw e;
		} catch (Throwable t) {
			throw new IllegalAcornStateException(t);
		}
	}
	
	@Override
	public String toString() {
	    StringBuilder sb = new StringBuilder();
	    sb.append("ClusterUpdateOperation [uid=").append(uid).append("] [info=").append(info).append("] [ccs=").append(ccs).append("] [chunk=").append("]");
	    return sb.toString();
	}
}