package org.simantics.acorn;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.simantics.db.Database;
import org.simantics.db.DatabaseUserAgent;
import org.simantics.db.Driver;
import org.simantics.db.ServerI;
import org.simantics.db.ServerReference;
import org.simantics.db.Session;
import org.simantics.db.SessionReference;
import org.simantics.db.exception.DatabaseException;

public class AcornDriver implements Driver {

    public static final String AcornDriverName = "acorn";

    private Map<String, ServerI> servers = new HashMap<>();
    private Map<String, Management> managements = new HashMap<>();
    
    @Override
    public String getName() {
        return AcornDriverName;
    }

    @Override
    public DatabaseUserAgent getDatabaseUserAgent(String address) throws DatabaseException {
        return AcornDatabaseManager.getDatabase(Paths.get(address)).getUserAgent();
    }

    @Override
    public void setDatabaseUserAgent(String address, DatabaseUserAgent dbUserAgent) throws DatabaseException {
        AcornDatabaseManager.getDatabase(Paths.get(address)).setUserAgent(dbUserAgent);
    }

    @Override
    public Session getSession(String address, Properties properties) throws DatabaseException {
        Path dbFolder = Paths.get(address);
        Session session = AcornSessionManagerImpl.getInstance().createSession(new SessionReference() {
            
            @Override
            public ServerReference getServerReference() {
                return new ServerReference() {
                    
                    @Override
                    public Path getDBFolder() {
                        return dbFolder;
                    }
                };
            }

            @Override
            public long getSessionId() {
                return 0L;
            }
        }, null);
        if (!properties.containsKey("clientId"))
            properties.put("clientId", dbFolder.toAbsolutePath().toString());
        session.registerService(Properties.class, properties);
        Session s = session.peekService(Session.class);
        if (null == s)
            session.registerService(Session.class, session);
        return session;
    }

    @Override
    public ServerI getServer(String address, Properties properties) throws DatabaseException {
        ServerI server = servers.get(address);
        if (server == null) {
            server = new AcornServerI(AcornDatabaseManager.getDatabase(Paths.get(address)), address);
            servers.put(address, server);
        }
        return server;
    }

    @Override
    public Management getManagement(String address, Properties properties) throws DatabaseException {
        Management mgmt = managements.get(address);
        if (mgmt == null) {
            mgmt = new AcornManagement(AcornDatabaseManager.getDatabase(Paths.get(address)), properties);
            managements.put(address, mgmt);
        }
        return mgmt;
    }
    
    private static class AcornServerI implements ServerI {
        
        private Database database;
        private String address;

        public AcornServerI(Database db, String address) {
            this.database = db;
            this.address = address;
        }
        
        @Override
        public void stop() throws DatabaseException {
            database.tryToStop();
        }
        
        @Override
        public void start() throws DatabaseException {
            database.start();
        }
        
        @Override
        public boolean isActive() throws DatabaseException {
            return database.isRunning();
        }
        
        @Override
        public String getAddress() throws DatabaseException {
            return address;
        }
        
        @Override
        public String executeAndDisconnect(String command) throws DatabaseException {
            return "";
        }
        
        @Override
        public String execute(String command) throws DatabaseException {
            return "";
        }
    }

}
