package org.simantics.migration.ui;

import gnu.trove.map.hash.THashMap;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.db.layer0.util.Simantics;
import org.simantics.db.request.Read;
import org.simantics.layer0.SoftwareConfigurationResources;

public class MigrationAnalysis {
    
    public class Update implements Comparable<Update> {
        Resource updateResource;
        Resource parentVersion;
        Resource childVersion;
        String versionName;
        
        public Update(Resource updateResource, Resource parentVersion,
                Resource childVersion, String versionName) {
            this.updateResource = updateResource;
            this.parentVersion = parentVersion;
            this.childVersion = childVersion;
            this.versionName = versionName;
        }

        @Override
        public int compareTo(Update o) {
            return versionName.compareTo(o.versionName);
        }

        public void execute(final Resource target) throws DatabaseException {
            Update parentUpdate = updates.get(parentVersion);
            if(parentUpdate != null)
                parentUpdate.execute(target);
            
            Collection<Runnable> migrateActions = Simantics.getSession().syncRequest(new Read<Collection<Runnable>>() {
                @Override
                public Collection<Runnable> perform(ReadGraph graph)
                        throws DatabaseException {
                    SoftwareConfigurationResources SC = SoftwareConfigurationResources.getInstance(graph);
                    ArrayList<Runnable> result = new ArrayList<Runnable>(); 
                    for(Resource action : graph.getObjects(updateResource, SC.Update_HasMigrationAction))
                        result.add(graph.adapt(action, ActionFactory.class).create(target));
                    return result;
                }                
            });
            for(Runnable action : migrateActions)
                action.run();
        }
        
        public void updateTargetVersion(final Resource target) throws DatabaseException {
            Simantics.getSession().syncRequest(new WriteRequest() {                        
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    graph.markUndoPoint();
                    SoftwareConfigurationResources SC = SoftwareConfigurationResources.getInstance(graph);
                    graph.deny(target, SC.IsCompatibleWith, MigrationAnalysis.this.parentVersion);
                    graph.claim(target, SC.IsCompatibleWith, childVersion);
                }
            });
        }
    }
    
    Resource parentVersion;
    THashMap<Resource, Update> updates = new THashMap<Resource, Update>();

    public MigrationAnalysis(ReadGraph g, Resource parentVersion) throws DatabaseException {
        this.parentVersion = parentVersion;
        updates.put(parentVersion, null);
        findUpdatesFrom(g, parentVersion);
        updates.remove(parentVersion);
    }

    private void findUpdatesFrom(ReadGraph g, Resource version) throws DatabaseException {        
        SoftwareConfigurationResources SC = SoftwareConfigurationResources.getInstance(g);
        for(Resource update : g.getObjects(version, SC.Version_HasUpdate)) {
            Resource to = g.getSingleObject(update, SC.Update_To);
            if(!updates.contains(to)) {
                String name = NameUtils.getSafeLabel(g, to);
                updates.put(to, new Update(update, version, to, name));
                findUpdatesFrom(g, to);
            }
        }
    }
    
    public Collection<Update> getUpdates() {
        return updates.values();
    }
}
