/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.utils;

import java.util.Arrays;
import java.util.List;

import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.utils.datastructures.Callback;

/**
 * @author Tuukka Lehtonen
 */
public class TagUtil {

    private final String  virtualGraphId;
    private final String  tagURI;
    private final boolean tag;

    public static void execute(Session session, String virtualGraphId, String tagURI, boolean tag, Resource... resources) {
        execute(session, virtualGraphId, tagURI, tag, Arrays.asList(resources));
    }

    public static void execute(Session session, String virtualGraphId, String tagURI, boolean tag, List<Resource> resources) {
        new TagUtil(virtualGraphId, tagURI, tag).execute(session, resources);
    }

    public TagUtil(String virtualGraphId, String tagURI, boolean tag) {
        this.virtualGraphId = virtualGraphId;
        this.tagURI = tagURI;
        this.tag = tag;
    }

    public void execute(Session session, Resource... resources) {
        execute(session, Arrays.asList(resources));
    }

    public void execute(Session session, final List<Resource> resources) {
        VirtualGraph vg = virtualGraphId == null ? null :
            session.getService(VirtualGraphSupport.class).getWorkspacePersistent(virtualGraphId);

        session.asyncRequest(new WriteRequest(vg) {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                processSelection(graph, resources);
            }
        }, new Callback<DatabaseException>() {
            @Override
            public void run(DatabaseException e) {
                if (e != null)
                    handleError(e);
            }
        });
    }

    public void syncExecute(Session session, Resource... resources) throws DatabaseException {
        syncExecute(session, Arrays.asList(resources));
    }

    public void syncExecute(Session session, final List<Resource> resources) throws DatabaseException {
        VirtualGraph vg = virtualGraphId == null ? null :
            session.getService(VirtualGraphSupport.class).getWorkspacePersistent(virtualGraphId);
        session.syncRequest(new WriteRequest(vg) {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                processSelection(graph, resources);
            }
        });
    }

    protected void processSelection(WriteGraph graph, List<Resource> resources) throws DatabaseException {

    	if (tagURI == null)
            return;
    	
    	final Resource tr = graph.getResource(tagURI);
        for (final Resource r : resources) {
            if (tag) {
            	// If r is virtual, we perform tagging in the same vg
            	if(virtualGraphId == null) {
            		VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
            		VirtualGraph vg = vgs.getGraph(graph, r);
            		if(vg != null) {
            			graph.sync(new WriteRequest(vg) {

							@Override
							public void perform(WriteGraph graph) throws DatabaseException {
		                        graph.claim(r, tr, r);
		                        CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
		                        graph.addMetadata(cm.add(ObjectUtils.toString("Marking " + NameUtils.getSafeLabel(graph, r) + " to " + NameUtils.getSafeName(graph, tr))));
							}
            				
            			});
            		} else {
                        graph.claim(r, tr, r);
                        CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                        graph.addMetadata(cm.add(ObjectUtils.toString("Marking " + NameUtils.getSafeLabel(graph, r) + " to " + NameUtils.getSafeName(graph, tr))));            		}
            	} else {
                    graph.claim(r, tr, r);
                    CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                    graph.addMetadata(cm.add(ObjectUtils.toString("Marking " + NameUtils.getSafeLabel(graph, r) + " to " + NameUtils.getSafeName(graph, tr))));
            	}
            } else {
                graph.deny(r, tr, r);
                CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                graph.addMetadata(cm.add(ObjectUtils.toString("Denying " + NameUtils.getSafeLabel(graph, r) + " to " + NameUtils.getSafeName(graph, tr))));
            }
        }
        
    }

    protected void handleError(DatabaseException e) {
        Logger.defaultLogError(e);
    }

}
