/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.utils;

import java.util.Properties;

import org.apache.log4j.Level;
import org.simantics.databoard.Bindings;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.common.internal.config.InternalClientConfig;
import org.simantics.utils.Development;

public class Logger {
    public static final boolean ENABLED = true;
    //public static final boolean ECHO = Development.DEVELOPMENT || false;
    public static final Properties defaultProperties = new Properties();
    static {
        defaultProperties.put("log4j.rootCategory", "ERROR, default");
        defaultProperties.put("log4j.appender.default", "org.apache.log4j.FileAppender");
        defaultProperties.put("log4j.appender.default.File", InternalClientConfig.DB_CLIENT_LOG_FILE);
        defaultProperties.put("log4j.appender.default.append", "true");
        defaultProperties.put("log4j.appender.default.layout", "org.apache.log4j.PatternLayout");
        defaultProperties.put("log4j.appender.default.layout.ConversionPattern", "%d{ISO8601} %-6r [%15.15t] %-5p %30.30c - %m%n");
    }
    private static LogManager defaultLogManager = new LogManager(defaultProperties);
    private static final Logger defaultErrorLogger = new Logger(LogManager.class);
    private org.apache.log4j.Logger logger;
    Logger(Class<?> clazz) {
        logger = defaultLogManager.getLogger(clazz);
    }

    /**
     * Log a trace event.
     *
     * @param message message of the trace
     * @param exception the exception, or <code>null</code>
     */
    public void logTrace(String message, Throwable exception) {

        if(!logger.isTraceEnabled()) return;

        // Errors are much more useful with a stack trace!
        if (exception == null) {
            exception = new RuntimeException();
        }
        logger.trace(message, exception);

        if (Development.DEVELOPMENT) {
            if(Development.<Boolean>getProperty(DevelopmentKeys.LOGGER_ECHO, Bindings.BOOLEAN)) {
                System.err.println("Logger.logTrace: " + message);
            }
        }

    }

    /**
     * Log an info event.
     *
     * @param message message of the info
     * @param exception the exception, or <code>null</code>
     */
    public void logInfo(String message, Throwable exception) {

        if(!logger.isInfoEnabled()) return;

        // Errors are much more useful with a stack trace!
        if (exception == null) {
            exception = new RuntimeException();
        }
        logger.info(message, exception);

        if (Development.DEVELOPMENT) {
            if(Development.<Boolean>getProperty(DevelopmentKeys.LOGGER_ECHO, Bindings.BOOLEAN)) {
                System.err.println("Logger.logInfo: " + message);
            }
        }

    }

    /**
     * Log an error event.
     *
     * @param message message of the error
     * @param exception the exception, or <code>null</code>
     */
    public void logError(String message, Throwable exception) {

        if(!logger.isEnabledFor(Level.ERROR)) return;

        // Errors are much more useful with a stack trace!
        if (exception == null) {
            exception = new RuntimeException();
        }
        logger.error(message, exception);

        if (Development.DEVELOPMENT) {
            if(Development.<Boolean>getProperty(DevelopmentKeys.LOGGER_ECHO, Bindings.BOOLEAN)) {
                System.err.println("Logger.logError: " + message);
            }
        }

    }

    /**
     * Log an error event.
     *
     * @param message message of the error
     * @param exception the exception, or <code>null</code>
     */
    public void logWarning(String message, Throwable exception) {

        if(!logger.isEnabledFor(Level.WARN)) return;

        // Errors are much more useful with a stack trace!
        if (exception == null) {
            exception = new RuntimeException();
        }
        logger.error(message, exception);

        if (Development.DEVELOPMENT) {
            if(Development.<Boolean>getProperty(DevelopmentKeys.LOGGER_ECHO, Bindings.BOOLEAN)) {
                System.err.println("Logger.logWarning: " + message);
            }
        }

    }

    /**
     * Log message.
     *
     * @param message to log.
     */
    public void logMessage(String message) {
        Level level = logger.getLevel();
        boolean toggle = !logger.isInfoEnabled();
        if (toggle)
            logger.setLevel((Level)Level.INFO);
        logger.info(message);
        if (toggle)
            logger.setLevel(level);
        if (Development.DEVELOPMENT) {
            if(Development.<Boolean>getProperty(DevelopmentKeys.LOGGER_ECHO, Bindings.BOOLEAN)) {
                System.err.println("Logger.logMessage: " + message);
            }
        }
    }

    public static Logger getDefault() {
        return defaultErrorLogger;
    }

    public static LogManager getDefaultLogManager() {
        return defaultLogManager;
    }
    public static void defaultLogError(Throwable exception) {
        if(!ENABLED) return;
        getDefault().logError(exception.getLocalizedMessage(), exception);
    }
    public static void defaultLogError(String message) {
        if(!ENABLED) return;
        getDefault().logError(message, null);
    }
    public static void defaultLogError(String message, Throwable exception) {
        if(!ENABLED) return;
        getDefault().logError(message, exception);
    }
    public static void defaultLogInfo(String message) {
        if(!ENABLED) return;
        getDefault().logInfo(message, null);
    }
    public static void defaultLogTrace(String message) {
        if(!ENABLED) return;
        getDefault().logTrace(message, null);
    }
    public static void defaultLog(String message) {
        if(!ENABLED) return;
        getDefault().logMessage(message);
    }
}
