/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.utils;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

/**
 * Utilities for exception propagation in the database client implementation.
 * 
 * @author Tuukka Lehtonen
 */
public final class ExceptionUtil {

    /**
     * Re-wrap and rethrow a previously caught exception.
     * 
     * <p>
     * Attempts to create a new instance of the specified throwable itself with
     * the specified Throwable as its cause.
     * 
     * Should the construction of the new throwable fail, the method will log
     * the error with {@link Logger} and just throw the originally
     * specified throwable itself.
     * 
     * @param <T> the concrete class of the throwable to be wrapped
     * @param t the throwable to wrap
     * @return never returns
     * @throws T the new wrapping throwable or the original throwable if the
     *         wrapping failed
     */
    public static <T extends Throwable> void wrapAndThrow(T t) throws T {
        try {
            @SuppressWarnings("unchecked")
            Class<T> clazz = (Class<T>) t.getClass();
            Constructor<T> ctor = clazz.getConstructor(Throwable.class);
            T parent = ctor.newInstance(t);
            throw parent;
        } catch (InstantiationException e1) {
            Logger.defaultLogError(e1);
            throw t;
        } catch (IllegalAccessException e1) {
            Logger.defaultLogError(e1);
            throw t;
        } catch (IllegalArgumentException e1) {
            Logger.defaultLogError(e1);
            throw t;
        } catch (InvocationTargetException e1) {
            Logger.defaultLogError(e1);
            throw t;
        } catch (SecurityException e1) {
            Logger.defaultLogError(e1);
            throw t;
        } catch (NoSuchMethodException e1) {
            Logger.defaultLogError(e1);
            throw t;
        }
    }

}
