/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.request;

import java.util.Collection;

import org.simantics.databoard.binding.Binding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.uri.ResourceToPossibleURI;
import org.simantics.db.common.uri.ResourceToURI;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;

/**
 * Facade class that aggregates the most common queries.
 * 
 * Example:
 * 
 *   System.out.println(  session.read( Queries.adapt(r, String.class) )  );
 *
 * @author Tuukka Lehtonen
 * @author Toni Kalajainen
 */
public class Queries {

    public static AsyncRead<Resource> resource(String uri) {
        return new org.simantics.db.common.primitiverequest.Resource(uri);
    }

    public static Read<String> name(final Resource resource) {
        return new ResourceRead<String>(resource) {
            @Override
            public String perform(ReadGraph graph) throws DatabaseException {
                return graph.getRelatedValue(resource, Layer0.getInstance(graph).HasName);
            }
        };
    }

    public static Read<String> safeName(final Resource resource) {
        return new ResourceRead<String>(resource) {
            @Override
            public String perform(ReadGraph graph) throws DatabaseException {
                return NameUtils.getSafeName(graph, resource);
            }
        };
    }

    public static Read<String> name(final String uri) {
        return new UnaryRead<String, String>(uri) {
            @Override
            public String perform(ReadGraph graph) throws DatabaseException {
                Resource resource = graph.getResource(uri);
                return graph.getRelatedValue(resource, Layer0.getInstance(graph).HasName);
            }
        };
    }

    public static Read<String> uri(Resource resource) {
        return new ResourceToURI(resource);
    }

    public static Read<String> possibleUri(Resource resource) {
        return new ResourceToPossibleURI(resource);
    }

    public static <T> Read<T> adapt(Resource resource, Class<T> target)
    {
        return new Adapt<T>(resource, target);
    }

    public static <T> Read<T> adapt(Resource resource, Class<T> target, boolean allowNull)
    {
        return new Adapt<T>(resource, target, allowNull);
    }

    public static <T> Read<T> adapt(Resource resource, Class<T> target, boolean allowNull, boolean uniqueResult)
    {
        return new Adapt<T>(resource, target, allowNull, uniqueResult);
    }

    public static Read<Boolean> isInstanceOf(Resource resource, Resource type)
    {
        return new IsInstanceOfQuery(resource, type);
    }

    public static Read<Boolean> isInstanceOf(Resource resource, String type)
    {
        return new IsInstanceOfQuery2(resource, type);
    }

    public static <T> Read<T> getRelatedValue(Resource subject, Resource relation, Binding binding)
    {
        return new ReadRelatedValue<T>(subject, relation, binding);
    }

    public static <T> Read<T> getRelatedValue(Resource subject, String relation, Binding binding)
    {
        return new ReadRelatedValue2<T>(subject, relation, binding);
    }

    public static <T> Read<T> getPossibleRelatedValue(Resource subject, Resource relation, Binding binding)
    {
        return new ReadPossibleRelatedValue<T>(subject, relation, binding);
    }
    
    public static Read<Boolean> hasTag(Resource subject, Resource tag) {
        return new HasTag(subject, tag);
    }

    public static ObjectsWithType objectsWithType(Resource subject, Resource subrelationOf, Resource instanceOf) {
        return new ObjectsWithType(subject, subrelationOf, instanceOf);
    }

	public static Read<Resource> possibleObjectWithType(final Resource resource, final Resource predicate, final Resource type) {
		return new PossibleObjectWithType(resource, predicate, type);
	}
	
	public static Read<Resource> possibleObject(final Resource subject, final Resource predicate) {
		return new Read<Resource>() {
			@Override
			public Resource perform(ReadGraph graph) throws DatabaseException {
				return graph.getPossibleObject(subject, predicate);
			}
		};		
	}
	
	public static Read<Collection<Resource>> objects(final Resource subject, final Resource predicate) {
		return new Read<Collection<Resource>>() {
			@Override
			public Collection<Resource> perform(ReadGraph graph) throws DatabaseException {
				return graph.getObjects(subject, predicate);
			}
		};		
	}	
	
    
}


class IsInstanceOfQuery implements Read<Boolean> {

    final protected Resource resource;
    final protected Resource type;

    public IsInstanceOfQuery(Resource resource, Resource type) {
        this.resource = resource;
        this.type = type;
    }

    @Override
    public int hashCode() {
        return resource.hashCode() + 13 * type.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (getClass() != object.getClass())
            return false;
        IsInstanceOfQuery other = (IsInstanceOfQuery) object;
        return resource.equals(other.resource) && type.equals(other.type);
    }

    @Override
    public Boolean perform(ReadGraph graph) throws DatabaseException {
        return graph.isInstanceOf(resource, type);
    }

}

class IsInstanceOfQuery2 implements Read<Boolean> {

    final protected Resource resource;
    final protected String type;

    public IsInstanceOfQuery2(Resource resource, String type) {
        this.resource = resource;
        this.type = type;
    }

    @Override
    public int hashCode() {
        return resource.hashCode() + 13 * type.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (getClass() != object.getClass())
            return false;
        IsInstanceOfQuery other = (IsInstanceOfQuery) object;
        return resource.equals(other.resource) && type.equals(other.type);
    }

    @Override
    public Boolean perform(ReadGraph graph) throws DatabaseException {
        Resource typeResource = graph.getResource(type);
        return graph.isInstanceOf(resource, typeResource);
    }

}


class ReadRelatedValue<T> implements Read<T> {

    Resource subject;
    Resource relation;
    Binding binding;

    public ReadRelatedValue(Resource subject, Resource relation, Binding binding) {
        this.subject = subject;
        this.relation = relation;
        this.binding = binding;
    }

    @Override
    public int hashCode() {
        return subject.hashCode() + 13 * relation.hashCode() + 7*binding.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (getClass() != object.getClass())
            return false;
        ReadRelatedValue<?> other = (ReadRelatedValue<?>) object;
        return subject.equals(other.subject) &&
        relation.equals(other.relation) &&
        binding.equals(other.binding);
    }

    @SuppressWarnings("unchecked")
    @Override
    public T perform(ReadGraph graph) throws DatabaseException {
        return (T) graph.getRelatedValue(subject, relation, binding);
    }
}


class ReadRelatedValue2<T> implements Read<T> {

    Resource subject;
    String relation;
    Binding binding;

    public ReadRelatedValue2(Resource subject, String relation, Binding binding) {
        this.subject = subject;
        this.relation = relation;
        this.binding = binding;
    }

    @Override
    public int hashCode() {
        return subject.hashCode() + 13 * relation.hashCode() + 7*binding.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (getClass() != object.getClass())
            return false;
        ReadRelatedValue2<?> other = (ReadRelatedValue2<?>) object;
        return subject.equals(other.subject) &&
        relation.equals(other.relation) &&
        binding.equals(other.binding);
    }

    @SuppressWarnings("unchecked")
    @Override
    public T perform(ReadGraph graph) throws DatabaseException {
        Resource relationResource = graph.getResource(relation);
        return (T) graph.getRelatedValue(subject, relationResource, binding);
    }
}

class ReadPossibleRelatedValue<T> implements Read<T> {

    Resource subject;
    Resource relation;
    Binding binding;

    public ReadPossibleRelatedValue(Resource subject, Resource relation, Binding binding) {
        this.subject = subject;
        this.relation = relation;
        this.binding = binding;
    }

    @Override
    public int hashCode() {
        return subject.hashCode() + 13 * relation.hashCode() + 7*binding.hashCode() + 5453;
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (getClass() != object.getClass())
            return false;
        ReadPossibleRelatedValue<?> other = (ReadPossibleRelatedValue<?>) object;
        return subject.equals(other.subject) &&
        relation.equals(other.relation) &&
        binding.equals(other.binding);
    }

    @SuppressWarnings("unchecked")
    @Override
    public T perform(ReadGraph graph) throws DatabaseException {
        return (T) graph.getPossibleRelatedValue(subject, relation, binding);
    }
}

class HasTag extends ResourceRead2<Boolean> {
    public HasTag(Resource subject, Resource tag) {
        super(subject, tag);
    }

    @Override
    public Boolean perform(ReadGraph graph) throws DatabaseException {
        return Boolean.valueOf(graph.hasStatement(resource, resource2, resource));
    }
}