/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.request;

import org.simantics.db.AsyncRequestProcessor;
import org.simantics.db.RequestProcessor;
import org.simantics.db.VirtualGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.request.DelayedWriteResult;
import org.simantics.db.request.UndoTraits;
import org.simantics.db.request.WriteInterface;
import org.simantics.db.request.WriteTraits;


/**
 * Usage samples:
 * 
 * <p>
 * Synchronous read without checked exceptions:
 * </p>
 * <pre>
 * Resource r = new SimpleReadGraphRequest&lt;Resource&gt;(session) {
 *     {@code @Override}
 *     public Resource run(Graph g) {
 *         Resource r = doSomethingThatMayFail();
 *         return r;
 *     }
 * }.sync();
 * </pre>
 * 
 * <p>
 * Synchronous read with a selected checked exception that may be thrown:
 * </p>
 * <pre>
 * Resource r = new SimpleReadGraphRequest&lt;Resource&gt;(session) {
 *     {@code @Override}
 *     public Resource run(Graph g) throws CheckedException {
 *         Resource r = doSomethingThatMayFailWithCheckedException();
 *         return r;
 *     }
 * }.sync(CheckedException.class);
 * </pre>
 * 
 * <p>
 * Asynchronous read with a completed() method for examining the result:
 * </p>
 * <pre>
 * Resource r = new SimpleReadGraphRequest&lt;Resource&gt;(session) {
 *     {@code @Override}
 *     public Resource run(Graph g) {
 *         Resource r = doSomethingThatMayFail();
 *         return r;
 *     }
 *     public void completed(Resource result) {
 *         // quickly dispatch the result somewhere, do nothing serious in this code.
 *     }
 * }.async();
 * </pre>
 * 
 * @param <T> the result type
 */
public abstract class DelayedWriteResultRequest<T> implements DelayedWriteResult<T>, WriteTraits, WriteInterface<T> {

    private final VirtualGraph provider;

    public DelayedWriteResultRequest() {
        this.provider = null;
    }

    public DelayedWriteResultRequest(VirtualGraph provider) {
        this.provider = provider;
    }
    
    @Override
    final public VirtualGraph getProvider() {
        return provider;
    }
    
    @Override
    public UndoTraits getUndoTraits() {
    	return null;
    }

    @Override
    public void request(AsyncRequestProcessor processor, final Procedure<T> procedure) {
    	processor.asyncRequest(this, procedure);
    }
    
    @Override
    public T request(RequestProcessor processor) throws DatabaseException {
    	return processor.syncRequest(this);
    }
    
}
