/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl.contribution;

import java.util.Collections;
import java.util.Map;

import org.eclipse.jface.resource.ImageDescriptor;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.BuiltinKeys.ImagerKey;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.content.Imager;
import org.simantics.browsing.ui.graph.impl.request.ResourceQuery;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.PendingVariableException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.Procedure;
import org.simantics.utils.datastructures.Callback;
import org.simantics.utils.ui.ErrorLogger;

public abstract class FinalImagerContributionImpl implements Imager {

    private final static Map<String, ImageDescriptor>         PENDING = Collections.emptyMap();

    protected final PrimitiveQueryUpdater                     updater;
    private final ResourceQuery<Map<String, ImageDescriptor>> imagerQuery;
    
    protected final NodeContext                                 context;
    private final BuiltinKeys.ImagerKey                       key;

    protected Map<String, ImageDescriptor>                      content = null;

    public Object getIdentity(ImagerKey key) {
        return key;
    }

    public FinalImagerContributionImpl(final PrimitiveQueryUpdater updater, final NodeContext context, final ImagerKey key) {

        this.updater = updater;
        this.context = context;
        this.key = key;

        imagerQuery = new ResourceQuery<Map<String, ImageDescriptor>>(getIdentity(key), context) {

            @Override
            public Map<String, ImageDescriptor> perform(ReadGraph graph) throws DatabaseException {
                try {
                    return getDescriptors(graph, context);
                } catch (PendingVariableException e) {
                    return PENDING;
                } catch (DatabaseException e) {
                    throw e;
                } catch (Throwable t) {
                    ErrorLogger.defaultLogError("LazyGraphLabeler.labelQuery produced unexpected exception.", t);
                    return null;
                }
            }

            @Override
            public String toString() {
                return FinalImagerContributionImpl.this + " with context " + context;
            }

        };

    }

    protected Procedure<Map<String, ImageDescriptor>> createProcedure() {

        return new Procedure<Map<String, ImageDescriptor>>() {

            @Override
            public void execute(Map<String, ImageDescriptor> result) {
                replaceResult(result);
            }

            @Override
            public void exception(Throwable t) {
                ErrorLogger.defaultLogError("FinalImagerContributionImpl.imagerQuery failed, see exception for details.", t);
            }

        };

    }

    protected void replaceResult(Map<String, ImageDescriptor> desc) {
        content = desc;
        updater.scheduleReplace(context, key, this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public ImageDescriptor getImage(String key) {
    	
        if (content == null) {

            content = PENDING;

            final DataSource<ReadGraph> source = updater.getDataSource(ReadGraph.class);
            assert(source != null);

            final Procedure<Map<String, ImageDescriptor>> procedure = createProcedure();
            source.schedule(new Callback<ReadGraph>() {
                @Override
                public void run(ReadGraph source) {

                    if(procedure instanceof Listener<?>)
                        source.asyncRequest(imagerQuery, (Listener<Map<String, ImageDescriptor>>)procedure);
                    else
                        source.asyncRequest(imagerQuery, procedure);

                }
            });

            return null;

        }

        ImageDescriptor descriptor = content.get(key);
        return descriptor;
    }

    // OVERRIDE

    public abstract Map<String, ImageDescriptor> getDescriptors(ReadGraph graph, NodeContext context) throws DatabaseException;

}
