/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl.contribution;

import org.simantics.browsing.ui.BuiltinKeys.ImageDecoratorKey;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.content.ImageDecorator;
import org.simantics.browsing.ui.graph.impl.request.ResourceQuery;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.PendingVariableException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.Procedure;
import org.simantics.utils.datastructures.Callback;
import org.simantics.utils.ui.ErrorLogger;

public abstract class FinalImageDecoratorContributionImpl implements ImageDecorator {

    final ImageDecorator FRESH = new ImageDecorator.Stub();
    final ImageDecorator PENDING = new ImageDecorator.Stub();
    final ImageDecorator NO_DECORATION = new ImageDecorator.Stub();

    protected final PrimitiveQueryUpdater       updater;
    private final ResourceQuery<ImageDecorator> imageQuery;
    private final ImageDecoratorKey             key;

    /**
     * The current decorator result produced for this contribution. The value
     * should never be <code>null</code>, but one of {@link #FRESH},
     * {@link #PENDING}, {@link #NO_DECORATION} instead.
     */
    protected ImageDecorator                      decorator = FRESH;

    final protected NodeContext                   context;

    public Object getIdentity(ImageDecoratorKey key) {
        return key;
    }

    private void request() {

        final DataSource<ReadGraph> source = updater.getDataSource(ReadGraph.class);
        assert(source != null);

        final Procedure<ImageDecorator> procedure = createProcedure();
        
        source.schedule(new Callback<ReadGraph>() {
            @Override
            public void run(ReadGraph source) {

                if(procedure instanceof Listener<?>)
                    source.asyncRequest(imageQuery, (Listener<ImageDecorator>)procedure);
                else
                    source.asyncRequest(imageQuery, procedure);

            }
        });

    }

    @Override
    public <Image> Image decorateImage(Image image, String column, int itemIndex) {
        if(FRESH == decorator) {
            decorator = PENDING;
            request();
        }
        return decorator.decorateImage(image, column, itemIndex);
    }

    public FinalImageDecoratorContributionImpl(final PrimitiveQueryUpdater updater, final NodeContext context, final ImageDecoratorKey key) {

        this.updater = updater;
        this.context = context;
        this.key = key;

        imageQuery = new ResourceQuery<ImageDecorator>(getIdentity(key), context) {

            @Override
            public ImageDecorator perform(ReadGraph graph) throws DatabaseException {
                try {
                    return getDecorator(graph, context);
                } catch (PendingVariableException e) {
                    return PENDING;
                } catch (DatabaseException e) {
                    throw e;
                } catch (Throwable t) {
                    ErrorLogger.defaultLogError("FinalImageDecoratorContributionImpl.imageQuery produced unexpected exception.", t);
                    return null;
                }
            }

            @Override
            public String toString() {
                return FinalImageDecoratorContributionImpl.this + " with context " + context;
            }

        };
        
    }

    protected Procedure<ImageDecorator> createProcedure() {

        return new Procedure<ImageDecorator>() {

            @Override
            public void execute(ImageDecorator result) {
                replaceResult(result);
            }

            @Override
            public void exception(Throwable t) {
                ErrorLogger.defaultLogError("FinalImageDecoratorContributionImpl.imageQuery failed, see exception for details.", t);
            }

        };

    }

    protected void replaceResult(ImageDecorator result) {
        // Never let decorator become null, use a stub decorator instead.
        if (result == null)
            result = NO_DECORATION;

        decorator = result;
        updater.scheduleReplace(context, key, this);
    }

    // OVERRIDE

    public abstract ImageDecorator getDecorator(ReadGraph graph, NodeContext context) throws DatabaseException;

}
